import { FilteredOrderItemType, OrderItemType } from "@/@types/order";
import { StatusType } from "@/@types/status";
import { clsx, type ClassValue } from "clsx";
import { twMerge } from "tailwind-merge";

export function cn(...inputs: ClassValue[]) {
  return twMerge(clsx(inputs));
}

export function toEnglishDigits(str: string) {
  return str.replace(/[۰-۹]/g, (d) => String("۰۱۲۳۴۵۶۷۸۹".indexOf(d)));
}

export function BankNameFaHandler(bank: string): string {
  const bankNameMap: Record<string, string> = {
    Melli: "بانک ملی",
    Mellat: "بانک ملت",
    Saderat: "بانک صادرات",
    Tejarat: "بانک تجارت",
    Sepah: "بانک سپه",
    Keshavarzi: "بانک کشاورزی",
    Maskan: "بانک مسکن",
    Refah: "بانک رفاه",
    Pasargad: "بانک پاسارگاد",
    Parsian: "بانک پارسیان",
    Saman: "بانک سامان",
    Eghtesad_Novin: "بانک اقتصاد نوین",
    Shahr: "بانک شهر",
    Karafarin: "بانک کارآفرین",
    Ayandeh: "بانک آینده",
    Sanat_Madan: "بانک صنعت و معدن",
    Sina: "بانک سینا",
    Iran_Zamin: "بانک ایران زمین",
    Sarmayeh: "بانک سرمایه",
    Melall: "بانک ملل",
    Resalat: "بانک رسالت",
    Postbank: "پست بانک",
    Khavar_Mianeh: "بانک خاورمیانه",
    Ansar: "بانک انصار",
    Ghavamin: "بانک قوامین",
    Hekmat: "بانک حکمت",
    Kosar: "بانک کوثر",
    Mehr_Eghtesad: "بانک مهر اقتصاد",
    Mehr_Iran: "بانک مهر ایران",
    Noor: "بانک نور",
    Bank_Markazi: "بانک مرکزی",
    Caspian: "بانک کاسپین",
    Dey: "بانک دی",
    Gardeshgari: "بانک گردشگری",
    Iran_Europe: "بانک ایران اروپا",
    Iran_Venezuela: "بانک ایران ونزوئلا",
    Standard_Chartered: "بانک استاندارد چارترد",
    Futurebank: "بانک فیوچر",
    Taavon_Eslami: "بانک تعاون اسلامی",
    Tosee: "بانک توسعه",
    Tosee_Saderat: "بانک توسعه صادرات",
    Tosee_Taavon: "بانک توسعه تعاون",
  };

  return bankNameMap[bank];
}

export function getStatusLabel(status: StatusType): string {
  const statusLabels: Record<StatusType, string> = {
    pending: "در انتظار تایید",
    approved: "تایید شده",
    rejected: "رد شده",
  };
  return statusLabels[status];
}

export const orderHandler = (data: OrderItemType[], order: OrderItemType) => {
  const cuurentData: OrderItemType[] = [...data];
  const index = cuurentData.findIndex(
    (item) => item.productid === order.productid
  );
  if (index === -1) {
    cuurentData.push(order);
  } else {
    cuurentData[index].count =
      Number(cuurentData[index].count) + Number(order.count);
  }
  return cuurentData;
};

export const isProductInCart = (order: OrderItemType[] | [], id: string) => {
  return order.some((item) => item.productid === id);
};
export const productCounter = (order: OrderItemType[] | [], id: string) => {
  const prodcutData = order.find((item) => item.productid === id);
  if (prodcutData) {
    return prodcutData.count;
  } else {
    return 0;
  }
};
export const addProductOrder = (
  orders: OrderItemType[],
  newOrder: OrderItemType
) => {
  const index = orders.findIndex(
    (order) => order.productid === newOrder.productid
  );
  if (index === -1) {
    return [...orders, newOrder];
  } else {
    orders[index].count = newOrder.count;
    orders[index].selectedOptions = newOrder?.selectedOptions;
    return orders;
  }
};

export const removeProductOrder = (
  orders: OrderItemType[],
  newOrder: OrderItemType
) => {
  const index = orders.findIndex(
    (order) => order.productid === newOrder.productid
  );
  if (index === -1) {
    return [...orders];
  } else {
    return orders.filter((item) => item?.productid !== newOrder.productid);
  }
};

export const increaseCountOrder = (
  orders: OrderItemType[],
  newOrder: OrderItemType
) => {
  const index = orders.findIndex(
    (order) => order.productid === newOrder.productid
  );
  if (index === -1) {
    return [...orders];
  } else {
    orders[index].count = newOrder.count;
    return [...orders];
  }
};
export const decreaseCountOrder = (
  orders: OrderItemType[],
  newOrder: OrderItemType
) => {
  const index = orders.findIndex(
    (order) => order.productid === newOrder.productid
  );
  if (index === -1) {
    return [...orders];
  } else {
    orders[index].count = newOrder.count;
    return [...orders];
  }
};

export const selectedOptionsHandler = (
  selectedOptions: {
    _id: string;
    title: string;
    value: number | string;
  }[],
  newOtion: {
    _id: string;
    title: string;
    value: number | string;
  }
) => {
  let options: {
    _id: string;
    title: string;
    value: number | string;
  }[] = selectedOptions;
  const hasOtion = selectedOptions.find((item) => item?._id === newOtion._id);
  if (!!hasOtion) {
    options = selectedOptions.filter((item) => item?._id !== newOtion._id);
  } else {
    options = [...selectedOptions, newOtion];
  }
  return options;
};

export const totalPrice = (orders: OrderItemType[]) => {
  return orders.reduce((acc, order) => {
    const basePrice = order.price * order.count;

    const optionsPrice =
      order.selectedOptions?.reduce(
        (sum, option) => sum + Number(option.value),
        0
      ) || 0;

    return acc + basePrice + optionsPrice;
  }, 0);
};

export function formatToPersianDate(isoDate: string): string {
  const date = new Date(isoDate);

  const parts = new Intl.DateTimeFormat("fa-IR-u-ca-persian", {
    weekday: "long",
    day: "numeric",
    month: "long",
    year: "numeric",
  }).formatToParts(date);

  const weekday = parts.find((p) => p.type === "weekday")?.value || "";
  const day = parts.find((p) => p.type === "day")?.value || "";
  const month = parts.find((p) => p.type === "month")?.value || "";
  const year = parts.find((p) => p.type === "year")?.value || "";

  return `${weekday} ${day} ${month} ${year}`;
}

export const findProductFromOrders = (orders: OrderItemType[], id: string) => {
  let product: null | OrderItemType = null;
  if (orders.length > 0) {
    const index = orders.findIndex((item) => item.productid === id);
    product = orders[index];
  }
  return product;
};
export function hasOption(
  options: {
    _id: string;
    title: string;
    value: number | string;
  }[],
  optionId: string
) {
  const isOption = options?.find((item) => item._id === optionId);
  return isOption;
}
export function hasProductsInOrders(orders: OrderItemType[], id: string) {
  const isProduct = orders?.find((item) => item.productid === id);
  return isProduct;
}

export function filteredIdInSelectOptions(orders: OrderItemType[]) {
  let products: FilteredOrderItemType[] | null = null;
  if (orders.length > 0) {
    products = orders.map((item) => ({
      ...item,
      selectedOptions: item.selectedOptions?.map((item) => ({
        title: item?.title,
        value: item?.value,
      })),
    }));
  }
  return products;
}

export function getMembershipDuration(dateString: string): string {
  const joinDate = new Date(dateString);
  const now = new Date();

  if (isNaN(joinDate.getTime())) return "تاریخ نامعتبر است";

  const diffMs = now.getTime() - joinDate.getTime();

  const years = diffMs / (1000 * 60 * 60 * 24 * 365);

  if (years < 1) {
    return "کمتر از یک سال عضویت";
  }

  const formattedYears = Number(years.toFixed(1));

  return `بیشتر از ${formattedYears} سال عضویت`;
}
export function buildSearchParams(
  ...objects: Record<string, unknown>[]
): string {
  const params = new URLSearchParams();

  objects.forEach((obj) => {
    Object.entries(obj).forEach(([key, value]) => {
      if (
        value !== "" &&
        value !== false &&
        value !== null &&
        value !== undefined &&
        !(typeof value === "number" && isNaN(value))
      ) {
        params.set(key, String(value));
      }
    });
  });

  const qs = params.toString();
  return qs ? `?${qs}` : "";
}
