"use client";

import React, { useState } from "react";

import { Button } from "@/components/ui/button";
import { ticketSchema } from "@/configs/ValidationSchema";
import { yupResolver } from "@hookform/resolvers/yup";
import { useForm } from "react-hook-form";
import { useMutation } from "@tanstack/react-query";
import { ApiError, ApiResponse } from "@/@types/api";
import { AxiosResponse } from "axios";
import { SendTicketData, TicketFormData } from "@/@types/tickets";
import { sendTicket } from "@/services/tickets";
import CustomToast from "@/components/modules/CustomToast";
import { useRouter } from "next/navigation";

export default function CreateTicketForm() {
  const router = useRouter();
  const {
    register,
    handleSubmit,
    formState: { errors },
  } = useForm<TicketFormData>({
    resolver: yupResolver(ticketSchema),
    defaultValues: {
      category: "",
      priority: "",
      text: "",
      title: "",
    },
  });

  const [fileError, setFileError] = useState<string | null>(null);
  const [file, setFile] = useState<File | null>(null);
  const [isFileError, setIsFileError] = useState<boolean>(false);

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    SendTicketData
  >({ mutationFn: sendTicket });

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0] ?? null;
    if (!selectedFile) return;

    if (selectedFile.size > 5 * 1024 * 1024) {
      setFileError("حجم فایل نباید بیشتر از 5MB باشد");
      setFile(null);
      return;
    }

    const allowedTypes = ["image/jpeg", "image/jpg", "image/png"];
    if (!allowedTypes.includes(selectedFile.type)) {
      setFileError("فرمت فایل باید jpg, jpeg, png باشد");
      setFile(null);
      return;
    }
    setFileError(null);
    setFile(selectedFile);
  };

  const submitHandler = (data: TicketFormData) => {
    if (fileError) {
      setIsFileError(true);
      return;
    }
    setIsFileError(false);

    mutate(
      {
        ...data,
        img: file ? file : "",
      },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "تیکت شما با موفقیت ارسال شد",
            });
          }
          router.push("/user-panel/tickets");
          router.refresh();
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 401) {
            CustomToast({
              type: "error",
              text: "دسترسی لازم برای ارسال تیکت برای شما وجود ندارد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };
  return (
    <form onSubmit={handleSubmit(submitHandler)}>
      <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-[33px]">
        <div className={"child:text-[#696464] dark:child:text-light"}>
          <p className="font text-lg">دسته بندی :</p>
          <select
            className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none"
            {...register("category")}
          >
            <option value="" disabled defaultChecked>
              دسته بندی سوال خود را انتخاب کنید
            </option>
            <option value="1">پشتیبانی محصولات</option>
            <option value="پشتیبانی">پشتیبانی</option>
            <option value="فنی"> فنی </option>
            <option value="امور مالی"> امور مالی </option>
            <option value="ثبت شکایت"> ثبت شکایت </option>
            <option value="ارتباط با مدیریت"> ارتباط با مدیریت </option>
            <option value="سایر"> سایر </option>
          </select>

          {errors.category && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">
                {errors.category?.message}
              </p>
            </div>
          )}
        </div>
        <div className={"child:text-[#696464] dark:child:text-light"}>
          <p className="font text-lg">میزان اولویت :</p>
          <select
            className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none"
            {...register("priority")}
          >
            <option value="" disabled defaultChecked>
              انتخاب کنید
            </option>
            <option value="زیاد">زیاد</option>
            <option value="متوسط">متوسط</option>
            <option value="3">کم</option>
          </select>

          {errors.priority && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">
                {errors.priority?.message}
              </p>
            </div>
          )}
        </div>
        <div className={"child:text-[#696464] dark:child:text-light"}>
          <p className="font text-lg"> موضوع تیکت :</p>
          <input
            className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
            type="text"
            placeholder="موضوع تیکت را وارد کنید"
            {...register("title")}
          />

          {errors.title && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">
                {errors.title?.message}
              </p>
            </div>
          )}
        </div>
      </div>
      <div
        className={
          "w-full mt-[30px] child:text-[#696464] dark:child:text-light"
        }
      >
        <p className="font text-lg"> پیام شما :</p>

        <textarea
          className="w-full mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-md outline-none overflow-hidden"
          placeholder="پیام خود را وارد کنید ...."
          {...register("text")}
          rows={6}
        />

        {errors.text && (
          <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
            <svg className="size-5 shrink-0">
              <use href="#danger"></use>
            </svg>
            <p className="text-wrap whitespace-normal">
              {errors.text?.message}
            </p>
          </div>
        )}
      </div>
      <div className="flex flex-col sm:flex-row items-center justify-between gap-y-7 mt-7 child:w-full sm:child:w-1/3">
        <div className="child:text-[#696464] dark:child:text-light">
          <p className="font text-lg">پیوست :</p>
          <label className="flex items-center justify-between w-full py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
            <span className="text-[#9B9B9B]">آپلود کنید</span>
            <input
              className="opacity-0 w-0 h-0"
              type="file"
              onChange={handleFileChange}
            />
            <svg className="w-6 h-6 text-[#9B9B9B]">
              <use href="#file"></use>
            </svg>
          </label>

          {isFileError && fileError && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">{fileError}</p>
            </div>
          )}
        </div>
        <div className="w-full mt-auto">
          <Button
            className="inline-block flex-center w-full h-[53px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded"
            type="submit"
            disabled={isPending}
          >
            ارسال پیام
          </Button>
        </div>
      </div>
    </form>
  );
}
