import { ApiError, ApiResponse } from "@/@types/api";
import { BankType } from "@/@types/Bank";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { bankShema } from "@/configs/ValidationSchema";
import { sendBank } from "@/services/bank";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React, { useState } from "react";
import { useForm } from "react-hook-form";
import BankComboBox from "./BankComboBox";

type AddBankCardModalProps = {
  setIsOpenModal: React.Dispatch<React.SetStateAction<boolean>>;
};
type input = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  value: "accountHolder" | "shaba" | "cardNumber";
  textarea?: boolean;
  ltr?: boolean;
  leftDir?: boolean;
};

export default function AddBankCardModal({
  setIsOpenModal,
}: AddBankCardModalProps) {
  const [selectValue, setSelectValue] = useState<string>("");
  const [bankError, setBankError] = useState<boolean>(false);

  const inputs: input[] = [
    {
      id: 1,
      label: " نام و نام خانوادگی :",
      required: false,
      inputType: "text",
      value: "accountHolder",
    },
    {
      id: 2,
      label: "  شماره شبا :",
      required: false,
      inputType: "text",
      value: "shaba",
      leftDir: true,
    },
    {
      id: 3,
      label: " شماره کارت :",
      required: false,
      inputType: "text",
      value: "cardNumber",
      ltr: true,
    },
  ];

  const queryClient = useQueryClient();
  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    BankType
  >({ mutationFn: sendBank });
  const {
    register,
    handleSubmit,
    formState: { errors },
  } = useForm<BankType>({
    resolver: yupResolver(bankShema),
    defaultValues: {
      accountHolder: "",
      shaba: "IR",
      cardNumber: "",
      bankName: "-1",
    },
  });

  const submitHandler = (data: BankType) => {
    if (!selectValue) {
      setBankError(true);
      return;
    }
    mutate(
      { ...data, bankName: selectValue },
      {
        onSuccess: (res) => {
          if (res.status === 201) {
            CustomToast({
              type: "success",
              text: "مشخصات بانکی با موفقیت ثبت شد",
              des: "درخواست شما در حال برسی توسط کارشناسان سایت می باشد",
            });
            setIsOpenModal(false);
            queryClient.invalidateQueries({ queryKey: ["get-bankCard"] });
          }
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 401) {
            CustomToast({
              type: "error",
              text: "عدم دسترسی به ارسال دیتا",
            });
          } else if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "درخواست قبلاً ثبت شده یا ورودی نامعتبر",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };

  return (
    <div className="fixed inset-0 flex items-center justify-center mx-auto w-screen h-screen overflow-hidden animate-fadeIn">
      <div className="w-[350px] md:w-[600px] p-3 bg-light dark:bg-dark-400 rounded-10 z-50">
        <div>
          <div className="flex items-center justify-between">
            <p className="font-bold text-base text-[#001A46] dark:text-light">
              افزودن کارت بانکی
            </p>
            <span
              className="inline-flex items-center justify-center size-7 rounded-sm bg-error-400 hover:bg-error-400/90 text-light duration-150 cursor-pointer"
              onClick={() => setIsOpenModal(false)}
            >
              X
            </span>
          </div>
          <div className="flex items-center mt-4">
            <span className="w-[20%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[80%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <form className="w-full" onSubmit={handleSubmit(submitHandler)}>
          <div className="grid  grid-cols-1 xl:grid-cols-2 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
            {inputs.map((input) => (
              <div
                key={input.id}
                className={"child:text-[#696464] dark:child:text-light"}
              >
                <p className="font text-lg">{input.label}</p>

                <input
                  className={`w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm ${
                    input.ltr ? "input__ltr" : ""
                  } ${input.leftDir ? "input__left" : ""}`}
                  type={input.inputType}
                  placeholder="وارد کنید"
                  {...register(input.value)}
                />

                {errors[input.value] && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors[input.value]?.message}
                    </p>
                  </div>
                )}
              </div>
            ))}
            <BankComboBox
              selectValue={selectValue}
              bankError={bankError}
              setSelectValue={setSelectValue}
            />
          </div>
          <div className="w-full mt-[32px]">
            <Button
              className="inline-block flex-center w-1/2 h-[53px] px-5 mx-auto bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded"
              type="submit"
              disabled={isPending}
              onClick={() => {
                if (!selectValue) {
                  setBankError(true);
                }
              }}
            >
              ارسال
            </Button>
          </div>
        </form>
      </div>

      <div
        className="absolute  w-full h-full z-40"
        onClick={() => {
          setIsOpenModal(false);
        }}
      ></div>
    </div>
  );
}
