"use client";

import React, { useState } from "react";

import { devDiscountType } from "@/@types/devDiscount";
import CustomToast from "@/components/modules/CustomToast";
import { getDevDiscount } from "@/services/dev";
import { useQuery } from "@tanstack/react-query";
import LoadingBox from "@/components/modules/LoadingBox";
import DeleteModal from "./DeleteModal";
import EditModal from "./EditModal";

export default function DiscountTable() {
  const [isOpenDeleteModal, setIsOpenDeleteModal] = useState<boolean>(false);
  const [isOpenEditModal, setIsOpenEditModal] = useState<boolean>(false);
  const [discountId, setDiscountId] = useState<string>("");
  const [discount, setDiscount] = useState<devDiscountType | null>(null);
  const { data: discounts, isLoading } = useQuery({
    queryKey: ["devDiscount"],
    queryFn: getDevDiscount,
    select: (res) => res.data,
  });

  const deleteHandler = (id: string) => {
    setIsOpenDeleteModal(true);
    setDiscountId(id);
  };
  const editHandler = (discount: devDiscountType) => {
    setIsOpenEditModal(true);
    setDiscountId(discount._id);
    setDiscount({
      _id: discount._id,
      code: discount.code,
      amount: Number(discount.amount),
      expiry: discount.expiry,
      usageLimit: Number(discount.usageLimit),
      usedCount: Number(discount.usedCount),
      fileid: discount.fileid,
    });
  };
  const copyHandler = async (copyValue: string) => {
    try {
      await navigator.clipboard.writeText(copyValue);
      CustomToast({ type: "success", text: "کد تخفیف کپی شد" });
    } catch {
      CustomToast({ type: "error", text: "خطا" });
    }
  };
  if (isLoading) return <LoadingBox />;

  return (
    <>
      <table className="w-full mt-7">
        <thead>
          <tr className="child:text-center child:text-[#A5A5A5] child:text-sm">
            <th>کد تخفیف</th>
            <th>میزان تخفیف</th>
            <th>تاریخ انقضا</th>
            <th>تعداد کل</th>
            <th>تعداد استفاده</th>
            <th>محصول</th>
            <th>عملیات</th>
          </tr>
        </thead>

        <tbody className="divide-y divide-dark/[7%] dark:divide-light/[6%]">
          {discounts.item.map((discount: devDiscountType) => (
            <tr
              key={discount._id}
              className=" child:py-[30px] child:px-2 child:text-center child:grow child:min-w-[100px] lg:child:w-[5%]"
            >
              <td>
                <p
                  className="text-center text-base cursor-pointer"
                  onClick={() => copyHandler(discount.code)}
                >
                  {discount.code}
                </p>
              </td>
              <td>
                <p className="text-[#858585] dark:text-light/80 font-medium text-sm">
                  % {discount.amount}
                </p>
              </td>
              <td className="text-wrap whitespace-normal text-[#1D1C1A] dark:text-light">
                <p className="text-center text-base">
                  {new Date(discount.expiry).toLocaleDateString("fa-IR")}
                </p>
              </td>
              <td className="">
                <p className="text-[#858585] dark:text-light/80 font-medium text-sm">
                  {discount.usageLimit}
                </p>
              </td>
              <td className="">
                <p className="text-[#858585] dark:text-light/80 font-medium text-sm">
                  {discount.usedCount}
                </p>
              </td>
              <td className="">
                <p className="text-[#858585] dark:text-light/80 font-medium text-sm">
                  {discount.fileid}
                </p>
              </td>
              <td>
                <div className="flex items-center justify-center gap-2.5">
                  <span
                    className="inline-block p-1.75 bg-[#3D96D1]/[12%] hover:bg-[#3D96D1]/20 dark:bg-[#3D96D1] dark:hover:bg-[#3D96D1]/80 duration-150 rounded cursor-pointer"
                    onClick={() => copyHandler(discount.code)}
                  >
                    <svg className="size-4 text-[#3B93E3] dark:text-light shrink-0">
                      <use href="#document-duplicate"></use>
                    </svg>
                  </span>
                  <span
                    className="inline-block p-1.75 bg-[#FF9D00]/[12%] hover:bg-[#FF9D00]/20 dark:bg-[#FF9D00] dark:hover:bg-[#FF9D00]/80 duration-150 rounded cursor-pointer"
                    onClick={() => editHandler(discount)}
                  >
                    <svg className="size-4 text-[#FF9D00] dark:text-light shrink-0">
                      <use href="#pencil-square"></use>
                    </svg>
                  </span>
                  <span
                    className="inline-block p-1.75 bg-[#EE404C]/[33%] hover:bg-[#EE404C]/40 dark:bg-[#EE404C] dark:hover:bg-[#EE404C]/80  duration-150 rounded cursor-pointer"
                    onClick={() => deleteHandler(discount._id)}
                  >
                    <svg className="size-4 text-[#E04F1A] dark:text-light shrink-0">
                      <use href="#trash"></use>
                    </svg>
                  </span>
                </div>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
      {isOpenEditModal && (
        <EditModal
          setIsOpenDeleteModal={setIsOpenEditModal}
          discountId={discountId}
          discount={discount}
        />
      )}
      {isOpenDeleteModal && (
        <DeleteModal
          setIsOpenDeleteModal={setIsOpenDeleteModal}
          discountId={discountId}
        />
      )}
    </>
  );
}
