"use client";

import { ApiError, ApiResponse } from "@/@types/api";
import { DevType } from "@/@types/dev";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { devInfoSchema } from "@/configs/ValidationSchema";
import { getDevInfo, updateDevInfo } from "@/services/dev";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import Link from "next/link";
import React, { useEffect } from "react";
import { useForm } from "react-hook-form";
type input = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  value: "title" | "bio" | "website" | "email";
  textarea?: boolean;
};
export default function DevInformation() {
  const queyClient = useQueryClient();

  const { data: devData } = useQuery({
    queryKey: ["devInfo"],
    queryFn: getDevInfo,
    select: (res) => res.data,
  });

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    DevType
  >({ mutationFn: updateDevInfo });

  const {
    register,
    handleSubmit,
    formState: { errors, isDirty },
    reset,
  } = useForm<DevType>({
    resolver: yupResolver(devInfoSchema),
  });

  useEffect(() => {
    if (devData?.dev) {
      reset({
        title: devData?.dev.title,
        bio: devData?.dev.bio,
        website: devData?.dev.website,
        // phone: devData?.dev.phone,
        email: devData?.dev.email,
      });
    }
  }, [devData, reset]);
  const inputs: input[] = [
    {
      id: 1,
      label: " عنوان :",
      required: false,
      inputType: "text",
      value: "title",
    },
    {
      id: 2,
      label: " وبسایت :",
      required: false,
      inputType: "text",
      value: "website",
    },
    // {
    //   id: 4,
    //   label: "  شماره موبایل :",
    //   required: false,
    //   inputType: "text",
    //   value: "phone",
    // },
    {
      id: 4,
      label: " ایمیل :",
      required: false,
      inputType: "email",
      value: "email",
    },
    {
      id: 5,
      label: "  معرفی فروشگاه :",
      required: false,
      inputType: "text",
      value: "bio",
      textarea: true,
    },
  ];
  const submitHandler = (data: DevType) => {
    if (!isDirty) {
      CustomToast({
        type: "error",
        text: "هیج تغییری در مشخصات کاربر انجام نشده است",
      });
      return;
    }
    mutate(data, {
      onSuccess: (res) => {
        if (res.status === 200) {
          CustomToast({
            type: "success",
            text: "مشخصات با موفقیت ثبت شد",
          });
          queyClient.invalidateQueries({ queryKey: ["devInfo"] });
        }
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 401) {
          CustomToast({
            type: "error",
            text: "عدم دسترسی به ارسال دیتا",
          });
        } else if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "درخواست قبلاً ثبت شده یا ورودی نامعتبر",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };
  return (
    <section className="w-full mt-6 py-6 px-[29px] bg-light dark:bg-[#393939]">
      <form onSubmit={handleSubmit(submitHandler)}>
        <div>
          <div className="flex items-center justify-between">
            <p className="font-bold text-sm lg:text-xl text-[#001A46] dark:text-light">
              مشخصات ثبت شده توسعه دهنده
            </p>
            {devData?.dev.status === "approved" && devData?.dev.isAccepted && (
              <Link
                className="inline-block flex-center h-11 px-5 bg-secondary-default hover:bg-secondary-default/90 text-xs-mines lg:text-base text-light rounded"
                href="developers/discount"
              >
                ایجاد کد تخفیف
              </Link>
            )}
          </div>
          <div className="flex items-center mt-4">
            <span className="w-[40%] lg:w-[25%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[60%] lg:w-[75%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
          {inputs.map((input) => (
            <div
              key={input.id}
              className={`child:text-[#696464] dark:child:text-light ${
                input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
              }`}
            >
              <p className="font text-lg">
                {input.label}{" "}
                {input.required ? (
                  <span className="text-error-400">*</span>
                ) : (
                  ""
                )}
              </p>
              {input.textarea ? (
                <textarea
                  className="w-full h-auto mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none  overflow-hidden"
                  placeholder="وارد کنید"
                  {...register(input.value)}
                  rows={4}
                />
              ) : (
                <input
                  className={`w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm ${
                    devData?.dev.status === "pending"
                      ? "!text-[#696464]/50 dark:!text-light/40"
                      : ""
                  }`}
                  type={input.inputType}
                  placeholder="وارد کنید"
                  {...register(input.value)}
                  disabled={devData?.dev.status === "pending"}
                />
              )}

              {errors[input.value] && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">
                    {errors[input.value]?.message}
                  </p>
                </div>
              )}
            </div>
          ))}
          <div className="w-full mt-auto">
            {devData?.dev.status === "rejected" ? (
              <Button
                className="inline-block flex-center w-full h-[53px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base text-light rounded"
                type="submit"
                disabled={isPending}
              >
                ویرایش اطلاعات
              </Button>
            ) : (
              <Button
                className={`inline-block flex-center w-full h-[53px] px-5 text-base dark:text-light rounded ${
                  devData?.dev.status === "pending"
                    ? "bg-primary-700 hover:bg-primary-700/80"
                    : "bg-secondary-default hover:bg-secondary-default/90"
                }`}
                type="submit"
                disabled={isPending || devData?.dev.status === "pending"}
              >
                {devData?.dev.status === "pending" && "درحال برسی مشخصات"}
                {devData?.dev.status === "approved" && " ذخیره تغییرات"}
              </Button>
            )}
          </div>
        </div>
      </form>
    </section>
  );
}
