import { OrderCountContext } from "@/app/context/OrderCountProvider";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { Checkbox } from "@/components/ui/Checkbox";
import {
  addProductOrder,
  decreaseCountOrder,
  findProductFromOrders,
  hasOption,
  hasProductsInOrders,
  isProductInCart,
  productCounter,
  removeProductOrder,
  selectedOptionsHandler,
} from "@/lib/utils";
import { useRouter } from "next/navigation";

import React, { RefObject, useContext, useEffect, useState } from "react";
import OrderModal from "./OrderModal";

type ProductDetailsProps = {
  price: number;
  features: { _id: string; name: string; detail: string }[];
  customOptions: { title: string; value: string | number; _id: string }[];
  productId: string;
  buyProductRef: RefObject<HTMLDivElement>;
  userName: string;
  salesCount: number;
  productUid: string;
  productName: string;
};

type SelectOptionType = {
  _id: string;
  title: string;
  value: number | string;
};

export default function ProductDetails({
  price,
  features,
  customOptions,
  productId,
  buyProductRef,
  userName,
  salesCount,
  productUid,
  productName,
}: ProductDetailsProps) {
  const counterContextData = useContext(OrderCountContext);
  const router = useRouter();
  const orderCount = productCounter(
    counterContextData?.localDate ?? [],
    productId
  );
  const [productCont, setProductCount] = useState(orderCount);

  const [productSelectedOptions, setProductsSelectedOptions] = useState<
    [] | SelectOptionType[]
  >([]);
  const [isOpenOrderModal, setIsOpenOrderModal] = useState<boolean>(false);

  useEffect(() => {
    const product = findProductFromOrders(
      counterContextData?.localDate ?? [],
      productId
    );
    if (product) {
      setProductsSelectedOptions(product?.selectedOptions ?? []);
    } else {
      setProductsSelectedOptions(customOptions ?? []);
    }
  }, [counterContextData?.localDate, productId, customOptions]);

  const addOrder = () => {
    const data = {
      productid: productId,
      price,
      count: orderCount || 1,
      selectedOptions: productSelectedOptions,
    };
    const addProduct = addProductOrder(
      counterContextData?.localDate ?? [],
      data
    );
    setProductCount(1);
    localStorage.setItem("orders", JSON.stringify(addProduct));
    counterContextData?.setLocalDate([...addProduct]);
    setIsOpenOrderModal(true);
  };

  const increaseOrder = (orderCount: number) => {
    const data = {
      productid: productId,
      price,
      count: orderCount,
      selectedOptions: productSelectedOptions,
    };
    const addProduct = addProductOrder(
      counterContextData?.localDate ?? [],
      data
    );

    counterContextData?.setLocalDate([...addProduct]);
    localStorage.setItem("orders", JSON.stringify(addProduct));
    CustomToast({
      text: "محصول مورد نظر با موفقیت به سبد خرید اضافه شد",
      type: "success",
    });
  };
  const removeOrder = () => {
    const data = { productid: productId, price, count: productCont };
    const removeProduct = removeProductOrder(
      counterContextData?.localDate ?? [],
      data
    );
    setProductCount(1);
    counterContextData?.setLocalDate([...removeProduct]);
    localStorage.setItem("orders", JSON.stringify(removeProduct));
    CustomToast({
      text: "محصول مورد نظر با موفقیت از سبد خرید حذف شد",
      type: "success",
    });
  };

  const decreaseOrder = (orderCount: number) => {
    const data = {
      productid: productId,
      price,
      count: orderCount,
      selectedOptions: productSelectedOptions,
    };
    const removeProduct = decreaseCountOrder(
      counterContextData?.localDate ?? [],
      data
    );

    counterContextData?.setLocalDate([...removeProduct]);
    localStorage.setItem("orders", JSON.stringify(removeProduct));
    CustomToast({
      text: "محصول مورد نظر با موفقیت از سبد خرید حذف شد",
      type: "success",
    });
  };

  const updateOrder = (data: {
    productid: string;
    price: number;
    count: number;
    selectedOptions: {
      _id: string;
      title: string;
      value: number | string;
    }[];
  }) => {
    const addProduct = addProductOrder(
      counterContextData?.localDate ?? [],
      data
    );
    counterContextData?.setLocalDate([...addProduct]);
    localStorage.setItem("orders", JSON.stringify(addProduct));
    CustomToast({
      text: "امکانات سفارشی محصول مورد نظر با موفقیت ویرایش شد",
      type: "success",
    });
  };

  const checkHandler = (option: {
    _id: string;
    title: string;
    value: number | string;
  }) => {
    const newOptions: {
      _id: string;
      title: string;
      value: number | string;
    }[] = selectedOptionsHandler(productSelectedOptions ?? [], option);
    setProductsSelectedOptions(newOptions);
    if (hasProductsInOrders(counterContextData?.localDate ?? [], productId)) {
      updateOrder({
        productid: productId,
        price,
        count: productCont,
        selectedOptions: newOptions,
      });
    }
  };

  return (
    <div className="w-full px-[22px] py-[19px] bg-light dark:bg-[#393939] border border-[#E4E4E4] dark:border-[#4E4E4E] rounded-lg">
      <div className="flex lg:flex-col xl:flex-row items-start justify-between gap-4">
        {/* //right */}
        <div>
          <div className="flex items-center  w-full">
            <p className="text-sm-mines font-medium mb-1.75">فروش محصول :</p>
            <span className="inline-block w-[30px] lg:w-[57px] h-px bg-[#EAEAEA]"></span>
          </div>
          <div className="flex items-center">
            <div className="w-[46px] h-10 bg-[#FFFC31]/30 dark:bg-light/30 py-1.75 px-2.5 rounded-10">
              <svg className="size-[26px]">
                <use href="#shop"></use>
              </svg>
            </div>
            <div className="flex-center px-[17px] py-1 mr-1 bg-[#FFFC31] border border-[#83817E] rounded-10">
              <p className="font-peyda font-extrabold text-sm lg:text-xl text-[#1D1C1A]">
                {salesCount}+
              </p>
            </div>
          </div>
        </div>
        {/* //left */}
        <div>
          <div className="flex items-center w-full">
            <p className="text-sm-mines font-medium mb-1.75">قیمت محصول :</p>
            <span className="inline-block w-[30px] lg:w-[90px] h-px bg-[#EAEAEA]"></span>
          </div>
          <div className="flex-center px-[17px] py-1 mr-1 bg-secondary-default rounded-10">
            <p className="font-peyda font-extrabold text-sm lg:text-xl text-light">
              {price?.toLocaleString() ?? 0}
            </p>
            <p className="mr-2.5 text-sm lg:text-base font-medium text-light">
              تومان
            </p>
          </div>
        </div>
      </div>
      {features.length > 0 &&
        features?.map((feature) => (
          <div
            key={feature._id}
            className="flex items-center gap-3.75 px-2.75 py-1.75 my-[25px] bg-[#FBFBFB] dark:bg-[#525252] border border-[#F5F5F5] dark:border-[#4E4E4E] rounded-md"
          >
            <svg className="size-5 text-secondary-default dark:!text-light shrink-0">
              <use href="#check-circle"></use>
            </svg>
            <p className="text-xs">
              {feature.name} {feature.detail}
            </p>
          </div>
        ))}
      {!!userName && (
        <Button
          className="hidden sm:inline-block flex-center w-full h-[55px] px-5 mt-1 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10 select-none"
          type="submit"
          onClick={() => router.push(`/store/${userName}`)}
        >
          مشاهده فروشگاه
        </Button>
      )}

      <div className="pt-3.5 border-t border-[#F5F5F5] dark:border-[#5E5E5E]">
        {customOptions?.length > 0 && (
          <>
            <p className="text-dark dark:text-light text-sm-mines font-bold">
              امکانات سفارشی :
            </p>
            {customOptions?.map((option) => (
              <div
                key={option?._id}
                className="flex items-center justify-between py-[17px] px-[21px] mt-2.25 bg-[#FFE351]/[22%] dark:bg-[#585858] rounded-sm border border-dashed border-[#D9D9D9] dark:border-[#4E4E4E]"
              >
                <div className="flex items-center gap-2.5">
                  <Checkbox
                    className="bg-light dark:bg-[#393939]"
                    checked={!!hasOption(productSelectedOptions, option._id)}
                    onClick={() => checkHandler(option)}
                  />
                  <p className="text-dark dark:text-light text-sm font-medium">
                    {option?.title}
                  </p>
                </div>
                <p className="font-peyda font-extrabold text-[#3C3C3C] dark:text-light text-base-mines">
                  {option?.value}
                </p>
              </div>
            ))}
          </>
        )}

        {/* counter input */}
        {orderCount > 0 && (
          <div className="flex items-center justify-between w-1/2 py-2.5 px-3.5 my-2.5 mx-auto border border-[#E1E1E1] rounded-sm child:text-[#081035]">
            <svg
              className={`size-4 cursor-pointer ${
                productCont === 10 ? "!text-[#081035]/40" : ""
              }`}
              onClick={() => {
                if (productCont < 10) {
                  setProductCount((prev) => prev + 1);
                  increaseOrder(productCont + 1);
                }
              }}
            >
              <use href="#plus"></use>
            </svg>
            <span className="select-none font-bold">{productCont}</span>
            {productCont > 1 ? (
              <svg
                className={`size-4 cursor-pointer ${
                  productCont === 1 ? "!text-[#081035]/30" : ""
                }`}
                onClick={() => {
                  if (productCont > 1) {
                    setProductCount((prev) => prev - 1);
                    decreaseOrder(productCont + 1);
                  }
                }}
              >
                <use href="#mines"></use>
              </svg>
            ) : (
              <svg
                className={`size-4 cursor-pointer !text-red-500 hover:!text-red-600`}
                onClick={removeOrder}
              >
                <use href="#trash"></use>
              </svg>
            )}
          </div>
        )}

        {!isProductInCart(counterContextData?.localDate ?? [], productId) && (
          <div ref={buyProductRef}>
            <Button
              className="hidden sm:inline-block flex-center w-full h-[55px] px-5 mt-4 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10 select-none"
              type="submit"
              onClick={addOrder}
            >
              افزودن به سبد خرید
            </Button>
          </div>
        )}
      </div>

      {isOpenOrderModal && (
        <OrderModal
          setIsOpenOrderModal={setIsOpenOrderModal}
          productName={productName}
          productUid={productUid}
        />
      )}
    </div>
  );
}
