import React from "react";
import { Button } from "@/components/ui/button";
import Image from "next/image";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import { phoneNumberSchema } from "@/configs/ValidationSchema";
import { useMutation } from "@tanstack/react-query";
import { sendPhone } from "@/services/auth";
import CustomToast from "@/components/modules/CustomToast";
import { ApiError, ApiResponse } from "@/@types/api";
import { AxiosResponse } from "axios";
type Inputs = {
  phone: string;
};
type PhoneLoginProps = {
  setStep: React.Dispatch<React.SetStateAction<number>>;
  setIsLoginModal: React.Dispatch<React.SetStateAction<boolean>>;
  setPhone: React.Dispatch<React.SetStateAction<string>>;
};

export default function PhoneLogin({
  setStep,
  setIsLoginModal,
  setPhone,
}: PhoneLoginProps) {
  const {
    register,
    handleSubmit,
    formState: { errors },
  } = useForm<Inputs>({
    resolver: yupResolver(phoneNumberSchema),
  });
  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    Inputs
  >({ mutationFn: sendPhone });
  const loginHandler = handleSubmit((data: Inputs) => {
    mutate(data, {
      onSuccess: (res) => {
        setStep(1);
        setPhone(data.phone);
        if (res.status === 200) {
          CustomToast({ type: "success", text: "کد تأیید ارسال شد" });
        }
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "کاربری با این شماره موبایل یافت نشد",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  });
  return (
    <form
      className="fixed top-0 flex items-center justify-center mx-auto w-screen h-screen z-[100] overflow-x-hidden animate-fadeIn "
      onSubmit={loginHandler}
    >
      <div
        className="absolute w-full h-full z-40"
        onClick={() => setIsLoginModal(false)}
      ></div>
      <div className="w-[350px] md:w-[400px] p-2.5 mx-auto bg-light rounded-2xl z-50">
        <div className="mx-auto">
          <Image
            src={"/images/logo.svg"}
            alt="عکس لوگو سایت داکوبیت"
            width={100}
            height={60}
            className="w-[130px] mx-auto"
          />
          <h2 className=" my-6 text-neutral-950 font-peyda font-extrabold text-center text-lg ">
            ورود با موبایل
          </h2>
          <h3 className="font-medium text-base text-center  text-neutral-950">
            حساب کاربری ندارید؟
            <span
              className=" text-secondary-default hover:text-secondary-default/80 cursor-pointer"
              onClick={() => setStep(3)}
            >
              {" "}
              ثبت نام
            </span>{" "}
            کنید
          </h3>
        </div>
        <div className="w-full my-4 mx-auto ">
          <div className="flex items-center justify-between w-2/3 h-[50px] p-1.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className="input__ltr w-full h-full px-2.5 bg-inherit text-neutral-600 outline-none"
              type="text"
              placeholder="شماره موبایل"
              {...register("phone")}
            />
            <svg className="size-5 text-neutral-400">
              <use href="#phone"></use>
            </svg>
          </div>
          <div className="w-2/3 mt-2  mx-auto">
            <span className=" text-sm text-error-400">
              {errors?.phone && errors?.phone.message}
            </span>
          </div>
        </div>
        <p
          className=" mt-1.5 mb-4 text-xs text-secondary-default hover:text-secondary-default/80 text-center cursor-pointer"
          onClick={() => setStep(4)}
        >
          ورود با رمز عبور
        </p>
        <div className="w-full mx-auto text-center">
          <Button
            className="bg-secondary-default hover:bg-secondary-default/90 text-light border border-light-400 rounded"
            size="lg"
            type="submit"
            disabled={isPending}
          >
            ادامه
          </Button>
        </div>
      </div>
    </form>
  );
}
