"use client";

import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { otpNumberSchema } from "@/configs/ValidationSchema";
import { setAccessToken } from "@/lib/token";
import { sendOtp } from "@/services/auth";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import Image from "next/image";
import React from "react";
import { useForm } from "react-hook-form";
type OtpLoginProps = {
  phone: string;
  step: number;
  setStep: React.Dispatch<React.SetStateAction<number>>;
  setIsLoginModal: React.Dispatch<React.SetStateAction<boolean>>;
  isRegister: boolean;
};
export default function OtpLogin({
  phone,
  setStep,
  setIsLoginModal,
  isRegister,
}: OtpLoginProps) {
  type Inputs = {
    otp: string;
  };
  const {
    register,
    handleSubmit,
    formState: { errors },
  } = useForm<Inputs>({
    resolver: yupResolver(otpNumberSchema),
  });

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    { phone: string; otp: string }
  >({
    mutationFn: sendOtp,
  });
  const queryClient = useQueryClient();
  const loginHandler = handleSubmit((data: Inputs) => {
    mutate(
      {
        phone,
        otp: data.otp,
      },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: isRegister
                ? "ثبت نام با موفقیت انجام شد"
                : "با موفقیت وارد شدید",
            });
          }
          setAccessToken(res?.data?.data);
          queryClient.invalidateQueries({ queryKey: ["get-me"] });
          setIsLoginModal(false);
          setStep(0);
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 402) {
            CustomToast({
              type: "error",
              text: "شماره موبایل یا ایمیل قبلا توی سایت ثبت نام شده است",
            });
            setStep(3);
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
            setStep(3);
          }
        },
      }
    );
  });
  return (
    <form
      className="fixed top-0 flex items-center justify-center mx-auto w-screen h-screen !z-[100] overflow-x-hidden animate-fadeIn "
      onSubmit={loginHandler}
    >
      <div
        className="absolute w-full h-full z-40"
        onClick={() => setIsLoginModal(false)}
      ></div>
      <div className="w-[350px] md:w-[400px] p-2.5 mx-auto bg-light rounded-2xl z-50">
        <div className="mx-auto">
          <Image
            src={"/images/logo.svg"}
            alt="عکس لوگو سایت داکوبیت"
            width={100}
            height={60}
            className="w-[130px] mx-auto"
          />
          <h2 className="my-6 text-neutral-950 font-peyda font-extrabold text-center text-lg ">
            کد تایید را وارد کنید
          </h2>
          <h3 className="font-medium text-sm text-center  text-neutral-400">
            کد تایید به شماره موبایل <span className="font-bold">{phone}</span>{" "}
            ارسال شد.
            <span
              onClick={() => setStep(isRegister ? 3 : 0)}
              className="text-sm text-secondary-default hover:text-secondary-default/80 cursor-pointer"
            >
              {" "}
              ویرایش
            </span>{" "}
          </h3>
        </div>
        <div className="w-full my-4 mx-auto ">
          <div className="flex items-center justify-between w-2/3 h-[50px] py-1.5 px-2.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className="input__ltr w-full h-full  bg-inherit text-neutral-600 outline-none"
              type="text"
              placeholder="کد تایید"
              {...register("otp")}
            />
            <svg className="size-5 text-neutral-400">
              <use href="#chat-bubble-left-ellipsis"></use>
            </svg>
          </div>
          <div className="w-2/3 mt-2  mx-auto">
            <span className=" text-sm text-error-400">
              {errors?.otp && errors?.otp.message}
            </span>
          </div>
        </div>
        <div className="w-full mx-auto text-center">
          <Button
            className="bg-secondary-default hover:bg-secondary-default/90 text-light border border-light-400 rounded"
            size="lg"
            type="submit"
            disabled={isPending}
          >
            تایید
          </Button>
        </div>
      </div>
    </form>
  );
}
