"use client";

import React, { useState } from "react";

import { Button } from "@/components/ui/button";
import { getUsers, updateUsersRole } from "@/services/users";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { userInfoAdminType } from "@/@types/user";
import CustomToast from "@/components/modules/CustomToast";
import { ApiError, ApiResponse } from "@/@types/api";
import { AxiosResponse } from "axios";
import LoadingBox from "@/components/modules/LoadingBox";
import EditModal from "./EditModal";

export default function UserTable() {
  const [isOpenEditModal, setIsOpenEditModal] = useState<boolean>(false);
  const [userData, setUserData] = useState<userInfoAdminType | null>(null);
  const queryClient = useQueryClient();

  const { data, isLoading } = useQuery({
    queryKey: ["admin-users"],
    queryFn: getUsers,
    select: (res) => res.data,
  });

  const { mutate } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    string
  >({
    mutationFn: updateUsersRole,
  });

  const changeRoleHandler = (id: string) => {
    mutate(id, {
      onSuccess: () => {
        CustomToast({ type: "success", text: "نقش کاربر با موفقیت تغییر کرد" });
        queryClient.invalidateQueries({ queryKey: ["admin-users"] });
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 404) {
          CustomToast({
            type: "error",
            text: "متأسفانه کاربر یافت نشد",
          });
        } else if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "شناسه نامعتبر",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };

  const editHandler = (data: userInfoAdminType) => {
    setUserData(data);
    setIsOpenEditModal(true);
  };
  if (isLoading) return <LoadingBox />;
  return (
    <>
      <table className="w-full mt-7">
        <thead>
          <tr className="child:text-center child:text-[#A5A5A5] child:text-sm">
            <th>نام</th>
            <th>نام خانوادگی</th>
            <th>نام کاربری</th>
            <th>ایمیل</th>
            <th>شماره موبایل</th>
            <th>نقش</th>
            <th>تغییر نقش</th>
            <th>ویرایش</th>
          </tr>
        </thead>
        <tbody className="divide-y divide-dark/[7%] dark:divide-light/[6%]">
          {data?.users.map((user: userInfoAdminType) => (
            <tr
              key={user._id}
              className=" child:py-[30px] child:px-2 child:text-center child:grow child:min-w-[100px] lg:child:w-[12%]"
            >
              <td className="text-wrap whitespace-normal text-[#1D1C1A] dark:text-light">
                <p className="text-center text-base">{user.fname}</p>
              </td>
              <td className="text-wrap whitespace-normal text-[#1D1C1A] dark:text-light">
                <p className="text-center text-base">{user.lname}</p>
              </td>
              <td className="text-wrap whitespace-normal text-[#1D1C1A] dark:text-light">
                <p className="text-center text-base">{user.username}</p>
              </td>
              <td className="text-wrap whitespace-normal text-[#1D1C1A] dark:text-light">
                <p className="text-center text-base">{user.email}</p>
              </td>
              <td className="text-wrap whitespace-normal text-[#1D1C1A] dark:text-light">
                <p className="text-center text-base">{user.phone}</p>
              </td>
              <td className="text-wrap whitespace-normal text-[#1D1C1A] dark:text-light">
                <p className="text-center text-base">
                  {user.role === "ADMIN" && (
                    <span className="text-secondary-default">مدیر سیستم</span>
                  )}
                  {user.role === "USER" && (
                    <span className="text-success-400">کاربر</span>
                  )}
                  {user.role === "SEO" && (
                    <span className="text-primary-700">نویسنده</span>
                  )}
                </p>
              </td>
              <td>
                <Button
                  className="flex-center h-[48px] mx-auto bg-secondary-default hover:bg-secondary-default/90 text-xs text-light rounded"
                  onClick={() => changeRoleHandler(user._id)}
                >
                  {user.role === "ADMIN"
                    ? "تغییر به کاربر"
                    : "تغییر به مدیر سیستم"}
                </Button>
              </td>
              <td>
                <Button
                  className="flex-center h-[48px] mx-auto bg-primary-700 hover:bg-primary-700/90 text-xs text-light rounded"
                  onClick={() => editHandler(user)}
                >
                  ویرایش کاربر
                </Button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
      {isOpenEditModal && (
        <EditModal setIsOpen={setIsOpenEditModal} userData={userData} />
      )}
    </>
  );
}
