"use client";
import { Button } from "@/components/ui/button";
import { addUserSchema } from "@/configs/ValidationSchema";
import { yupResolver } from "@hookform/resolvers/yup";
import React, { useState } from "react";
import { Controller, useForm } from "react-hook-form";
import DatePicker, { DateObject } from "react-multi-date-picker";
import persian from "react-date-object/calendars/persian";
import persian_fa from "react-date-object/locales/persian_fa";
import gregorian from "react-date-object/calendars/gregorian";

import * as yup from "yup";
import { AxiosResponse } from "axios";
import { ApiError, ApiResponse } from "@/@types/api";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { addUser } from "@/services/users";
import { AddUserType } from "@/@types/user";
import CustomToast from "@/components/modules/CustomToast";

type InputsType = yup.InferType<typeof addUserSchema>;

type inputs = {
  label: string;
  required: boolean;
  inputType: string;
  textarea?: boolean;
  checkbox?: boolean;
  value:
    | "username"
    | "fname"
    | "lname"
    | "phone"
    | "codemeli"
    | "email"
    | "address"
    | "gender"
    | "datebirth"
    | "website"
    | "homephone"
    | "postalcode"
    | "province"
    | "city"
    | "password";
};

// role: "USER";
// isdev: boolean;
// wallet: 0;

export default function UserForm() {
  const [dateValue, setDateValue] = useState<DateObject | string>("");

  const queryClient = useQueryClient();

  const {
    register,
    handleSubmit,
    formState: { errors },
    control,
    reset,
  } = useForm<InputsType>({
    resolver: yupResolver(addUserSchema),
    defaultValues: {
      username: "",
      fname: "",
      lname: "",
      phone: "",
      codemeli: "",
      email: "",
      address: "",
      gender: "male",
      datebirth: "",
      password: "",
      website: "",
      homephone: "",
      postalcode: "",
      province: "",
      city: "",
    },
  });

  const inputs: inputs[] = [
    {
      label: " نام کاربری :",
      required: true,
      inputType: "text",
      value: "username",
    },
    {
      label: " نام :",
      required: true,

      inputType: "text",
      value: "fname",
    },
    {
      label: " نام خانوادگی :",
      required: true,

      inputType: "text",
      value: "lname",
    },
    {
      label: "  شماره موبایل :",
      required: true,

      inputType: "text",
      value: "phone",
    },
    {
      label: "کد ملی :",
      required: true,

      inputType: "text",
      value: "codemeli",
    },
    {
      label: "ایمیل :",
      required: true,

      inputType: "text",
      value: "email",
    },
    {
      label: "وبسایت :",
      required: false,
      inputType: "text",
      value: "website",
    },
    {
      label: "شماره تماس :",
      required: false,
      inputType: "text",
      value: "homephone",
    },
    {
      label: "کدپستی :",
      required: false,
      inputType: "text",
      value: "postalcode",
    },
    {
      label: "استان :",
      required: false,
      inputType: "text",
      value: "province",
    },
    {
      label: "شهر :",
      required: false,
      inputType: "text",
      value: "city",
    },
    {
      label: "رمز عبور :",
      required: true,

      inputType: "text",
      value: "password",
    },
    {
      label: " آدرس :",
      required: true,
      inputType: "text",
      value: "address",
    },
  ];

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    AddUserType
  >({ mutationFn: addUser });

  const submitHandler = (data: InputsType) => {
    const userData = {
      ...data,
      username: data.username ?? "",
      fname: data.fname ?? "",
      lname: data.lname ?? "",
      phone: data.phone ?? "",
      codemeli: data.codemeli ?? "",
      email: data.email ?? "",
      address: data.address ?? "",
      password: data.password ?? "",
      datebirth: data.datebirth ?? "",
      gender: data.gender ?? "male",
      website: data.website ?? "",
      homephone: data.homephone ?? "",
      postalcode: data.postalcode ?? "",
      province: data.province ?? "",
      city: data.city ?? "",
      role: "USER" as const,
      isdev: false,
      wallet: 0,
    };
    mutate(userData, {
      onSuccess: () => {
        CustomToast({
          type: "success",
          text: "کاربر جدید با موفقیت ایجاد شد",
        });
        queryClient.invalidateQueries({ queryKey: ["admin-users"] });
        reset({
          username: "",
          fname: "",
          lname: "",
          phone: "",
          codemeli: "",
          email: "",
          address: "",
          gender: "male",
          datebirth: "",
          password: "",
          website: "",
          homephone: "",
          postalcode: "",
          province: "",
          city: "",
        });
        setDateValue("");
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        const msg = (error?.response?.data as { message?: string })?.message;
        if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: msg || "فیلدها اشتباه است",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };

  return (
    <form onSubmit={handleSubmit(submitHandler)}>
      <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
        {inputs.map((input) => (
          <div
            key={input.value}
            className={`child:text-[#696464] dark:child:text-light ${
              input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
            }`}
          >
            {!input.checkbox ? (
              <p className="font text-lg">
                {input.label}{" "}
                {input.required ? (
                  <span className="text-error-400">*</span>
                ) : (
                  ""
                )}
              </p>
            ) : null}
            {input.textarea ? (
              <textarea
                className="w-full  mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none"
                placeholder="وارد کنید"
                {...register(input.value)}
                rows={4}
              />
            ) : (
              <input
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                type={input.inputType}
                placeholder="وارد کنید"
                {...register(input.value)}
              />
            )}

            {errors?.gender && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors?.gender?.message}
                </p>
              </div>
            )}
          </div>
        ))}

        <div className="child:text-[#696464] dark:child:text-light">
          <p className="font text-lg">
            <span className="font text-lg">جنسیت :</span>
          </p>

          <select
            className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-[#535353] p-4 pl-10 text-base-mines rounded-sm outline-none"
            {...register("gender")}
          >
            <option value="-1" defaultChecked>
              انتخاب کنید
            </option>

            <option value={"male"}>مرد</option>
            <option value={"female"}>زن</option>
          </select>
          {errors.gender && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">
                {errors.gender?.message}
              </p>
            </div>
          )}
        </div>
        <div className="child:text-[#696464] dark:child:text-light">
          <p className="font text-lg">تاریخ تولد :</p>
          <Controller
            control={control}
            name="datebirth"
            rules={{ required: true }}
            render={({ field: { onChange } }) => (
              <div className="w-full h-full">
                <div className="flex items-center justify-between w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none">
                  <DatePicker
                    value={dateValue}
                    format="YYYY/MM/DD"
                    style={{
                      background: "inherit",
                      width: "150px",
                      height: "53px",
                      padding: "16px 0px 16px 16px",
                    }}
                    className="!w-full"
                    onChange={(date) => {
                      if (date?.isValid) {
                        setDateValue(date);

                        const g = date.convert(gregorian);

                        const year = Number(g.year);
                        const month = Number(g.month) - 1;
                        const day = Number(g.day);

                        const jsDate = new Date(year, month, day);

                        if (isNaN(jsDate.getTime())) {
                          onChange("");
                          return;
                        }

                        const tehranDate = new Intl.DateTimeFormat("en-CA", {
                          timeZone: "Asia/Tehran",
                          year: "numeric",
                          month: "2-digit",
                          day: "2-digit",
                        })
                          .format(jsDate)
                          .replace(/\//g, "-");

                        const zonedString = `${tehranDate}T00:00:00+03:30`;

                        onChange(zonedString);
                      } else {
                        setDateValue("");
                        onChange("");
                      }
                    }}
                    placeholder="وارد کنید"
                    inputClass="datePicker_style"
                    calendar={persian}
                    locale={persian_fa}
                    onOpen={() => {
                      const input = document.querySelector(".datePicker_style");
                      if (input) {
                        (input as HTMLInputElement).readOnly = true;
                        (input as HTMLInputElement).inputMode = "none";
                      }
                    }}
                    onClose={() => {
                      const input = document.querySelector(".datePicker_style");
                      if (input) {
                        (input as HTMLInputElement).readOnly = true;
                        (input as HTMLInputElement).inputMode = "none";
                      }
                    }}
                  />
                  <svg className="w-5 h-5 text-tertiary-40 shrink-0">
                    <use href="#date"></use>
                  </svg>
                </div>
                {errors?.datebirth && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors?.datebirth?.message}
                    </p>
                  </div>
                )}
              </div>
            )}
          />
        </div>
      </div>

      <div className="w-full text-left xl:text-left mt-[23px]">
        <Button
          className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
          type="submit"
          disabled={isPending}
        >
          ایجاد
        </Button>
      </div>
    </form>
  );
}
