"use client";
import { ApiError, ApiResponse } from "@/@types/api";
import { EditUserFormType, userInfoAdminType } from "@/@types/user";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { editUserSchema } from "@/configs/ValidationSchema";
import { toEnglishDigits } from "@/lib/utils";
import { updateUserInfo } from "@/services/users";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React, { Dispatch, SetStateAction, useEffect, useState } from "react";
import { Controller, useForm } from "react-hook-form";
import DatePicker, { DateObject } from "react-multi-date-picker";
import persian from "react-date-object/calendars/persian";
import persian_fa from "react-date-object/locales/persian_fa";
import gregorian from "react-date-object/calendars/gregorian";
import { Checkbox } from "@/components/ui/Checkbox";

type input = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  checkbox?: boolean;
  value:
    | "username"
    | "fname"
    | "lname"
    | "phone"
    | "codemeli"
    | "email"
    | "address"
    | "password"
    | "website"
    | "homephone"
    | "postalcode"
    | "province"
    | "city"
    | "isdev";
  textarea?: boolean;
  ltr?: boolean;
  leftDir?: boolean;
};
type EditModalProps = {
  setIsOpen: Dispatch<SetStateAction<boolean>>;
  userData: userInfoAdminType | null;
};

export default function EditModal({ setIsOpen, userData }: EditModalProps) {
  const [dateValue, setDateValue] = useState<DateObject | string>("");

  const inputs: input[] = [
    {
      id: 0,
      label: "نام کاربری :",
      required: false,
      inputType: "text",
      value: "username",
    },
    {
      id: 1,
      label: "نام :",
      required: false,
      inputType: "text",
      value: "fname",
    },
    {
      id: 2,
      label: "نام خانوادگی :",
      required: false,
      inputType: "text",
      value: "lname",
    },
    {
      id: 3,
      label: "شماره موبایل :",
      required: false,
      inputType: "text",
      value: "phone",
    },
    {
      id: 4,
      label: "کدملی :",
      required: false,
      inputType: "text",
      value: "codemeli",
    },
    {
      id: 5,
      label: "ایمیل :",
      required: false,
      inputType: "text",
      value: "email",
    },
    {
      id: 7,
      label: "رمزعبور :",
      required: false,
      inputType: "text",
      value: "password",
    },
    {
      id: 8,
      label: "وبسایت :",
      required: false,
      inputType: "text",
      value: "website",
    },
    {
      id: 9,
      label: "تلفن ثابت :",
      required: false,
      inputType: "text",
      value: "homephone",
    },
    {
      id: 10,
      label: "کد پستی :",
      required: false,
      inputType: "text",
      value: "postalcode",
    },
    {
      id: 11,
      label: "استان :",
      required: false,
      inputType: "text",
      value: "province",
    },
    {
      id: 12,
      label: "شهر :",
      required: false,
      inputType: "text",
      value: "city",
    },
    {
      id: 13,
      label: "آدرس :",
      required: false,
      inputType: "text",
      value: "address",
      textarea: true,
    },
    {
      id: 14,
      label: "برنامه نویس",
      required: false,
      inputType: "text",
      checkbox: true,
      value: "isdev",
    },
  ];

  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
    control,
  } = useForm<EditUserFormType>({
    resolver: yupResolver(editUserSchema),
    defaultValues: {
      username: "",
      fname: "",
      lname: "",
      phone: "",
      codemeli: "",
      email: "",
      address: "",
      password: "",
      gender: "مرد",
      datebirth: "",
      website: "",
      homephone: "",
      postalcode: "",
      province: "",
      city: "",
      isAuthentication: false,
      isflag: false,
      isdev: false,
      role: "USER",
    },
  });
  useEffect(() => {
    if (userData) {
      reset({
        username: userData.username || "",
        fname: userData.fname || "",
        lname: userData.lname || "",
        phone: userData.phone || "",
        codemeli: userData.codemeli || "",
        email: userData.email || "",
        address: userData.address || "",
        password: userData.password || "",
        gender: userData.gender || "مرد",
        datebirth: userData.datebirth || "",
        website: userData.website || "",
        homephone: userData.homephone || "",
        postalcode: userData.postalcode || "",
        province: userData.province || "",
        city: userData.city || "",
        isAuthentication: userData.isAuthentication ?? false,
        isflag: userData.isflag ?? false,
        isdev: userData.isdev ?? false,
        role: userData?.role ?? "USER",
      });
    }
    if (userData?.datebirth) {
      const converDate = toEnglishDigits(userData.datebirth);
      const persianDate = new DateObject({
        date: converDate,
        calendar: gregorian,
        format: "YYYY-MM-DD",
      }).convert(persian);
      setDateValue(persianDate);
    } else {
      setDateValue("");
    }
  }, [userData, reset]);

  const queryClient = useQueryClient();
  const { mutate } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    { id: string; userInfo: EditUserFormType }
  >({
    mutationFn: updateUserInfo,
  });
  const formSubmit = (data: EditUserFormType) => {
    mutate(
      { id: userData?._id ?? "", userInfo: data },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "مشخصات کاربر با موفقیت آپدیت شد",
            });
          }
          queryClient.invalidateQueries({ queryKey: ["admin-users"] });
          setIsOpen(false);
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "ورودی نامعتبر یا اطلاعات تکراری (مثلاً phone یا email)",
            });
          } else if (statusCode === 401) {
            CustomToast({
              type: "error",
              text: "عدم دسترسی به ارسال دیتا",
            });
          } else if (statusCode === 404) {
            CustomToast({
              type: "error",
              text: "کاربر یافت نشد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };
  return (
    <div className="fixed inset-0 flex items-center justify-center mx-auto w-screen h-screen overflow-hidden animate-fadeIn">
      <div className="w-[350px] md:w-[400px]w-[350px] md:w-[50%] h-full md:h-[97%]  p-3 bg-light dark:bg-dark-400 rounded-10 z-50 overflow-auto">
        <div>
          <p className="font-bold text-xl text-[#001A46] dark:text-light">
            ویرایش کاربر
          </p>
          <div className="flex items-center mt-4">
            <span className="w-[25%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[75%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <form className="w-full p-2">
          <div className="grid  grid-cols-1 md:grid-cols-2 gap-x-16 gap-y-2">
            {inputs.slice(0, 5).map((input) => (
              <div
                key={input.id}
                className={`child:text-[#696464] dark:child:text-light ${
                  input.textarea ? "col-span-1 md:col-span-2" : ""
                }`}
                // className="child:text-[#696464] dark:child:text-light"
              >
                <p className="font text-lg">
                  {input.label}{" "}
                  {/* {input.required ? <span className="text-error-400">*</span> : ""} */}
                </p>
                {input.textarea ? (
                  <textarea
                    className="w-full mt-2 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none lg:resize-none overflow-hidden"
                    placeholder="وارد کنید"
                    {...register(input.value)}
                    rows={8}
                  />
                ) : (
                  <input
                    className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                    type={input.inputType}
                    placeholder="وارد کنید"
                    {...register(input.value)}
                  />
                )}

                {errors[input.value] && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors[input.value]?.message}
                    </p>
                  </div>
                )}
              </div>
            ))}
            <div className={`child:text-[#696464] dark:child:text-light`}>
              <p className="font text-lg">جنسیت</p>

              <select
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none"
                {...register("gender")}
              >
                <option value="مرد" defaultChecked>
                  مرد
                </option>
                <option value="زن"> زن </option>
              </select>

              {errors.gender && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">
                    {errors.gender?.message}
                  </p>
                </div>
              )}
            </div>
            {/* datebirth */}
            <div>
              <div className="child:text-[#696464] dark:child:text-light">
                <p className="font text-lg">تاریخ تولد :</p>
                <Controller
                  control={control}
                  name="datebirth"
                  rules={{ required: true }}
                  render={({ field: { onChange } }) => (
                    <div className="w-full h-full">
                      <div className="flex items-center justify-between w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none">
                        <DatePicker
                          value={dateValue}
                          format="YYYY/MM/DD"
                          style={{
                            background: "inherit",
                            width: "150px",
                            height: "53px",
                            padding: "16px 0px 16px 16px",
                          }}
                          className="!w-full"
                          onChange={(date) => {
                            if (date?.isValid) {
                              setDateValue(date);
                              const gregorianDate = date
                                .convert(gregorian)
                                .format("YYYY-MM-DD");
                              onChange(gregorianDate);
                            } else {
                              setDateValue("");
                              onChange("");
                            }
                          }}
                          placeholder="وارد کنید"
                          inputClass="datePicker_style"
                          calendar={persian}
                          locale={persian_fa}
                          onOpen={() => {
                            const input =
                              document.querySelector(".datePicker_style");
                            if (input) {
                              (input as HTMLInputElement).readOnly = true;
                              (input as HTMLInputElement).inputMode = "none";
                            }
                          }}
                          onClose={() => {
                            const input =
                              document.querySelector(".datePicker_style");
                            if (input) {
                              (input as HTMLInputElement).readOnly = true;
                              (input as HTMLInputElement).inputMode = "none";
                            }
                          }}
                        />
                        <svg className="w-5 h-5 text-tertiary-40 shrink-0">
                          <use href="#date"></use>
                        </svg>
                      </div>
                    </div>
                  )}
                />
              </div>
              {errors.datebirth && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:!text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">
                    {errors.datebirth?.message}
                  </p>
                </div>
              )}
            </div>
            {inputs.slice(5).map((input) => (
              <div
                key={input.id}
                className={`child:text-[#696464] dark:child:text-light ${
                  input.textarea ? "col-span-1 md:col-span-2" : ""
                }`}
              >
                {input?.checkbox ? null : (
                  <p className="font text-lg">{input.label} </p>
                )}
                {input.textarea ? (
                  <textarea
                    className="w-full mt-2 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none lg:resize-none overflow-hidden"
                    placeholder="وارد کنید"
                    {...register(input.value)}
                    rows={8}
                  />
                ) : input?.checkbox ? (
                  <Controller
                    name="isdev"
                    control={control}
                    render={({ field }) => (
                      <div className="flex items-center gap-2 h-full mt-5">
                        <Checkbox
                          className="size-6"
                          checked={field.value}
                          onCheckedChange={field.onChange}
                        />
                        <p className="text-lg">{input?.label}</p>
                      </div>
                    )}
                  />
                ) : (
                  <input
                    className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                    type={input.inputType}
                    placeholder="وارد کنید"
                    {...register(input.value)}
                  />
                )}

                {errors[input.value] && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors[input.value]?.message}
                    </p>
                  </div>
                )}
              </div>
            ))}
            <div className={`child:text-[#696464] dark:child:text-light`}>
              <p className="font text-lg">نقش کاربر</p>

              <select
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none"
                {...register("role")}
              >
                <option value="USER" defaultChecked>
                  کاربر
                </option>
                <option value="ADMIN"> مدیر سییتم </option>
                <option value="SEO"> نویسنده </option>
              </select>

              {errors.role && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">
                    {errors.role?.message}
                  </p>
                </div>
              )}
            </div>
          </div>
        </form>
        <div className="w-full text-right xl:text-left mt-[23px] child:my-1">
          <Button
            className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 ml-1 bg-error-400 hover:bg-error-400/90 text-base text-light hover:text-light rounded-10"
            variant="outline"
            onClick={() => setIsOpen(false)}
          >
            انصراف
          </Button>
          <Button
            className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
            type="submit"
            // disabled={isPending}
            onClick={handleSubmit(formSubmit)}
          >
            ارسال
          </Button>
        </div>
      </div>

      <div
        className="absolute w-full h-full z-40"
        onClick={() => {
          setIsOpen(false);
        }}
      ></div>
    </div>
  );
}
