"use client";

import React, { useState } from "react";

import { Button } from "@/components/ui/button";
import { ticketInputSchema } from "@/configs/ValidationSchema";
import { yupResolver } from "@hookform/resolvers/yup";
import { useForm } from "react-hook-form";
import CustomToast from "@/components/modules/CustomToast";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { ApiError, ApiResponse } from "@/@types/api";
import { AxiosResponse } from "axios";
import { sendAdminTicketMessage } from "@/services/tickets";
import { SendTicketMessage } from "@/@types/tickets";

type Inputs = {
  text: string;
};

export default function AdminTicketChatInput({
  ticketId,
}: {
  ticketId: string;
}) {
  const [fileError, setFileError] = useState<string | null>(null);
  const [file, setFile] = useState<File | null>(null);
  const [isFileError, setIsFileError] = useState<boolean>(false);

  const queryClient = useQueryClient();

  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm<Inputs>({
    resolver: yupResolver(ticketInputSchema),
  });

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    SendTicketMessage
  >({ mutationFn: sendAdminTicketMessage });

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0] ?? null;
    if (!selectedFile) return;

    if (selectedFile.size > 5 * 1024 * 1024) {
      setFileError("حجم فایل نباید بیشتر از 5MB باشد");
      setFile(null);
      return;
    }

    const allowedTypes = ["image/jpeg", "image/jpg", "image/png"];
    if (!allowedTypes.includes(selectedFile.type)) {
      setFileError("فرمت فایل باید jpg, jpeg, png باشد");
      setFile(null);
      return;
    }
    setFileError(null);
    setFile(selectedFile);
  };

  const sendTicketHandler = (data: Inputs) => {
    if (fileError) {
      setIsFileError(true);
      return;
    }
    setIsFileError(false);
    mutate(
      {
        ...data,
        ticketId,
        img: file ? file : "",
      },
      {
        onSuccess: () => {
          reset({ text: "" });
          queryClient.invalidateQueries({ queryKey: ["ticket", ticketId] });
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 401) {
            CustomToast({
              type: "error",
              text: "دسترسی لازم برای ارسال تیکت برای شما وجود ندارد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };
  return (
    <form className="w-full mt-10" onSubmit={handleSubmit(sendTicketHandler)}>
      <div>
        <textarea
          className="w-full bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none overflow-hidden"
          placeholder="پیام خود را بنویسید ..."
          {...register("text")}
          rows={6}
        />
        {errors.text && (
          <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
            <svg className="size-5 shrink-0">
              <use href="#danger"></use>
            </svg>
            <p className="text-wrap whitespace-normal">
              {errors.text?.message}
            </p>
          </div>
        )}
      </div>
      <div className="flex flex-col sm:flex-row items-center justify-between gap-5">
        <div>
          <label className="flex items-center justify-between w-full sm:w-[180px] py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
            <span className="text-[#9B9B9B]">آپلود پیوست</span>
            <input
              className="opacity-0 w-0 h-0"
              type="file"
              onChange={handleFileChange}
            />
            <svg className="w-6 h-6 text-[#9B9B9B]">
              <use href="#file"></use>
            </svg>
          </label>
          {isFileError && fileError && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">{fileError}</p>
            </div>
          )}
        </div>

        <div className="w-full sm:w-[180px]">
          <Button
            className="flex-center w-full h-[48px] px-5 mx-auto bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded"
            disabled={isPending}
          >
            ارسال
          </Button>
        </div>
      </div>
    </form>
  );
}
