"use client";
import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { categorySchema } from "@/configs/ValidationSchema";
import { sendSubSubCategory } from "@/services/category";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React, { useState } from "react";
import { useForm } from "react-hook-form";

type InputsType = {
  title: string;
  href: string;
};
type inputs = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  textarea?: boolean;
  checkbox?: boolean;
  value: "title" | "href";
};

type SubSubCategoryFormType = {
  catId: string;
  subId: string;
};

export default function SubSubCategoryForm({
  catId,
  subId,
}: SubSubCategoryFormType) {
  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm<InputsType>({
    resolver: yupResolver(categorySchema),
    defaultValues: {
      title: "",
      href: "",
    },
  });

  const [fileError, setFileError] = useState<string | null>(null);
  const [file, setFile] = useState<File[]>([]);
  const [isFileError, setIsFileError] = useState<boolean>(false);

  const queryClient = useQueryClient();
  const inputs: inputs[] = [
    {
      id: 1,
      label: " عنوان :",
      required: true,
      inputType: "text",
      value: "title",
    },
    {
      id: 2,
      label: "مسیر (url) :",
      required: true,
      inputType: "text",
      value: "href",
    },
  ];
  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    {
      catId: string;
      subId: string;
      data: FormData;
    }
  >({
    mutationFn: sendSubSubCategory,
  });

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const fileList = e.target.files;
    if (!fileList) return;
    const allowedTypes = ["image/jpeg", "image/png", "image/webp"];
    const selectedFiles = Array.from(fileList);
    const invalidFiles = selectedFiles.filter(
      (file) => !allowedTypes.includes(file.type)
    );

    if (invalidFiles.length > 0) {
      setFileError("فقط فایل‌های تصویری با فرمت JPG, PNG یا WEBP مجاز هستند");
      setIsFileError(true);
      setFile([]);
      e.target.value = "";
      return;
    }

    setFileError(null);
    setIsFileError(false);
    setFile(selectedFiles);
    e.target.value = "";
  };

  const submitHandler = (data: InputsType) => {
    if (file.length === 0) {
      setIsFileError(true);
      setFileError(" عکس دسته‌بندی خود را انتخاب کنید");
      return;
    }
    const formData = new FormData();

    Object.entries(data).forEach(([key, value]) => {
      formData.append(key, value as string);
    });

    file.forEach((f) => {
      formData.append("img", f);
    });

    mutate(
      { catId, subId, data: formData },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "دسته بندی جدید با موفقیت ایجاد شد",
            });
          }
          queryClient.invalidateQueries({
            queryKey: ["admin-sub-sub-category"],
          });
          reset({
            title: "",
            href: "",
          });
          setFile([]);
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "ورودی معتبر نمی باشد",
            });
          } else if (statusCode === 401) {
            CustomToast({
              type: "error",
              text: "عدم دسترسی به ارسال دیتا",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };

  return (
    <form onSubmit={handleSubmit(submitHandler)}>
      <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
        {inputs.map((input) => (
          <div
            key={input.id}
            className={`child:text-[#696464] dark:child:text-light ${
              input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
            }`}
          >
            <p className="font text-lg">
              {input.label}{" "}
              {input.required ? <span className="text-error-400">*</span> : ""}
            </p>

            <input
              className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
              type={input.inputType}
              placeholder="وارد کنید"
              {...register(input.value)}
            />

            {errors[input.value] && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors[input.value]?.message}
                </p>
              </div>
            )}
          </div>
        ))}
        <div>
          <p className="font text-lg child:text-[#696464] dark:child:text-light">
            <span className="font text-lg">فایل :</span>
          </p>
          <label className="flex items-center justify-between w-full py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
            <span className="text-[#9B9B9B]">آپلود پیوست</span>
            <input
              className="opacity-0 w-0 h-0"
              type="file"
              accept="image/*"
              onChange={handleFileChange}
            />
            <svg className="w-6 h-6 text-[#9B9B9B]">
              <use href="#file"></use>
            </svg>
          </label>
          {isFileError && fileError && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">{fileError}</p>
            </div>
          )}
        </div>
      </div>
      <div className="w-full text-left xl:text-left mt-[23px]">
        <Button
          className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
          type="submit"
          disabled={isPending}
        >
          ارسال
        </Button>
      </div>
    </form>
  );
}
