"use client";
import React, { Dispatch, SetStateAction, useState } from "react";
import { Button } from "@/components/ui/button";
import { discountSchema } from "@/configs/ValidationSchema";
import { yupResolver } from "@hookform/resolvers/yup";
import { Controller, useForm } from "react-hook-form";
import DatePicker, { DateObject } from "react-multi-date-picker";
import persian from "react-date-object/calendars/persian";
import persian_fa from "react-date-object/locales/persian_fa";
import gregorian from "react-date-object/calendars/gregorian";
import CustomToast from "@/components/modules/CustomToast";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import { ApiError, ApiResponse } from "@/@types/api";
import { discountType, UpdateDiscountType } from "@/@types/discount";
import { updateDiscount } from "@/services/discount";

type InputsType = {
  code: string;
  amount: string;
  expiry: string;
  usageLimit: string;
};
type inputs = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  textarea?: boolean;
  checkbox?: boolean;
  value: "code" | "amount" | "expiry" | "usageLimit";
};

type EditDiscountFormProps = {
  setIsOpen: Dispatch<SetStateAction<boolean>>;
  discountData: discountType | null;
};
export default function EditDiscountForm({
  setIsOpen,
  discountData,
}: EditDiscountFormProps) {
  const initialExpiry = discountData?.expiry
    ? new DateObject({
        date: discountData.expiry,
        calendar: persian,
        locale: persian_fa,
      })
    : null;
  const [expiryDate, setExpiryDate] = useState<DateObject | null>(
    initialExpiry
  );

  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
    control,
  } = useForm<InputsType>({
    resolver: yupResolver(discountSchema),
    defaultValues: {
      code: discountData?.code || "",
      amount: String(discountData?.amount) || "",
      usageLimit: String(discountData?.usageLimit) || "",
      expiry: discountData?.expiry,
    },
  });

  const queryClient = useQueryClient();

  const inputs: inputs[] = [
    {
      id: 1,
      label: " کد تخفیف :",
      required: true,
      inputType: "text",
      value: "code",
    },
    {
      id: 2,
      label: " میزان تخفیف :",
      required: true,
      inputType: "text",
      value: "amount",
    },
    {
      id: 4,
      label: " تعداد کل تخفیف :",
      required: true,
      inputType: "text",
      value: "usageLimit",
    },
  ];
  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    UpdateDiscountType
  >({
    mutationFn: updateDiscount,
  });

  const submitHandler = (data: InputsType) => {
    const iso = new DateObject().convert(gregorian).toDate().toISOString();
    mutate(
      {
        id: discountData?._id || "",
        code: data?.code,
        amount: +data.amount,
        usageLimit: +data.usageLimit,
        expiry: data.expiry || iso,
      },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "کد تخفیف با موفقیت ویراش شد",
            });
            reset({
              code: "",
              amount: "",
              usageLimit: "",
              expiry: "",
            });
            setExpiryDate(null);
            queryClient.invalidateQueries({ queryKey: ["admin-discounts"] });
          }
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 401) {
            CustomToast({
              type: "error",
              text: "عدم دسترسی به ارسال دیتا",
            });
          } else if (statusCode === 404) {
            CustomToast({
              type: "error",
              text: "کد تخفیف یافت نشد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };
  return (
    <div>
      <div className="grid  grid-cols-1 md:grid-cols-2 gap-5 mt-5">
        {inputs.map((input) => (
          <div
            key={input.id}
            className={`child:text-[#696464] dark:child:text-light ${
              input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
            }`}
          >
            <p className="font text-lg">
              {input.label}{" "}
              {input.required ? <span className="text-error-400">*</span> : ""}
            </p>

            <input
              className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
              type={input.inputType}
              placeholder="وارد کنید"
              {...register(input.value)}
            />

            {errors[input.value] && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors[input.value]?.message}
                </p>
              </div>
            )}
          </div>
        ))}
        <div className="child:text-[#696464] dark:child:text-light">
          <p className="font text-lg">تاریخ انقضا : </p>

          <Controller
            control={control}
            name="expiry"
            rules={{ required: true }}
            render={({ field: { onChange } }) => (
              <div className="w-full h-full">
                <div className="flex items-center justify-between w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none">
                  <DatePicker
                    value={expiryDate}
                    format="YYYY/MM/DD"
                    style={{
                      background: "inherit",
                      width: "150px",
                      height: "53px",
                      padding: "16px",
                    }}
                    className="!w-full"
                    onChange={(date) => {
                      if (!date) {
                        setExpiryDate(null);
                        onChange("");
                        return;
                      }

                      setExpiryDate(date as DateObject);
                      const g = (date as DateObject).convert(gregorian);

                      const year = g.year;
                      const month = g.month.number;
                      const day = g.day;

                      const utcMs = Date.UTC(year, month - 1, day, 0, 0, 0, 0);
                      const isoUtc = new Date(utcMs).toISOString();

                      onChange(isoUtc);
                    }}
                    placeholder="وارد کنید"
                    inputClass="datePicker_style"
                    calendar={persian}
                    locale={persian_fa}
                  />
                  <svg className="w-5 h-5 text-tertiary-40 shrink-0">
                    <use href="#date"></use>
                  </svg>
                </div>
                {errors.expiry && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors.expiry?.message}
                    </p>
                  </div>
                )}
              </div>
            )}
          />
        </div>
      </div>
      <div className="flex items-center justify-center w-full gap-x-2 mt-[23px]">
        <Button
          className="hidden sm:inline-flex flex-center py-6 px-5 bg-error-400 hover:bg-error-400/90 text-base dark:text-light rounded-10"
          onClick={() => setIsOpen(false)}
        >
          لغو
        </Button>
        <Button
          className="hidden sm:inline-flex flex-center py-6 px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
          disabled={isPending}
          onClick={handleSubmit(submitHandler)}
        >
          ویرایش
        </Button>
      </div>
    </div>
  );
}
