"use client";
import { ApiError, ApiResponse } from "@/@types/api";
import { CommentsAdminType, CommentType } from "@/@types/comment";
import CustomToast from "@/components/modules/CustomToast";
import LoadingBox from "@/components/modules/LoadingBox";
import { Button } from "@/components/ui/button";
import {
  acceptComment,
  addTopComment,
  getAdminComments,
  rejectComment,
} from "@/services/comment";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React, { useEffect, useState } from "react";
import AnswerCommentModal from "./AnswerCommentModal";
import { redirect } from "next/navigation";

type CommentsTableProps = {
  commentId: string | null;
};

export default function CommentsTable({ commentId }: CommentsTableProps) {
  const [isOpen, setIsOpen] = useState<boolean>(false);
  const [commentData, setCommentData] = useState<null | CommentType>(null);
  const [commentsData, setCommentsData] = useState<CommentsAdminType | null>(
    null
  );
  const [fieldId, setFieldID] = useState<string | null>(null);

  const queryClient = useQueryClient();

  const { data, isLoading } = useQuery({
    queryKey: ["admin-comments"],
    queryFn: getAdminComments,
    select: (res) => res.data,
  });

  useEffect(() => {
    if (!commentId) return redirect("/admin-panel/comments");
  }, [commentId]);

  useEffect(() => {
    if (data && commentId) {
      const foundComment: CommentsAdminType | null = data?.data?.find(
        (comment: CommentsAdminType) => comment?._id === commentId
      );
      setCommentsData(foundComment ?? null);
      setFieldID(foundComment ? foundComment.fileid?._id : null);
    }
  }, [data]);

  const { mutate: acceptComment_mutate } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    { fileid: string; id: string }
  >({
    mutationFn: acceptComment,
  });

  const { mutate: rejectComment_mutate } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    { fileid: string; id: string }
  >({
    mutationFn: rejectComment,
  });

  const { mutate: topComment_mutate } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    { fileid: string; id: string }
  >({
    mutationFn: addTopComment,
  });

  const acceptCommentHandler = (id: string, fileid: string) => {
    acceptComment_mutate(
      { id, fileid },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "کامنت مورد نظر با موفقیت تایید شد",
            });
          }
          queryClient.invalidateQueries({
            queryKey: ["admin-comments"],
          });
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "ورودی معتبر نمی باشد",
            });
          } else if (statusCode === 404) {
            CustomToast({
              type: "error",
              text: "کامنت یافت نشد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };

  const rejectCommentHandler = (id: string, fileid: string) => {
    rejectComment_mutate(
      { id, fileid },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "کامنت مورد نظر با موفقیت رد شد",
            });
          }
          queryClient.invalidateQueries({ queryKey: ["admin-comments"] });
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "ورودی معتبر نمی باشد",
            });
          } else if (statusCode === 404) {
            CustomToast({
              type: "error",
              text: "کامنت یافت نشد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };

  const topCommentHandler = (id: string, fileid: string) => {
    topComment_mutate(
      { id, fileid },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "کامنت مورد نظر با موفقیت به لیست نتایج و تجربیات اضافه شد",
            });
          }
          queryClient.invalidateQueries({ queryKey: ["admin-comments"] });
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "ورودی معتبر نمی باشد",
            });
          } else if (statusCode === 404) {
            CustomToast({
              type: "error",
              text: "کامنت یافت نشد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };

  if (isLoading) return <LoadingBox />;

  return (
    <>
      <table className="w-full mt-7">
        <thead>
          <tr className=" child:text-[#454545] dark:child:text-light/80 child:text-sm">
            <th className="pb-3.25 border-b border-dark/[7%] dark:border-light/[6%]">
              متن کامنت
            </th>
            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              تاریخ ایجاد
            </th>
            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              تاریخ ویرایش
            </th>

            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              وضعیت
            </th>
            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              کامنت برتر
            </th>
            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              مشاهده / پاسخ
            </th>
          </tr>
        </thead>
        {commentsData ? (
          <tbody className="divide-y divide-dark/[7%] dark:divide-light/[6%]">
            {commentsData?.comments?.map((comment: CommentType) => (
              <tr
                key={comment._id}
                className=" child:py-[30px] child:px-2 child:text-right child:grow child:min-w-[150px] lg:child:w-[15%]"
              >
                <td>
                  <div className="flex items-center gap-2 xl:gap-[18px]">
                    <p className="text-[#575757] dark:text-light/80 text-sm  line-clamp-1">
                      {comment?.text}
                    </p>
                  </div>
                </td>
                <td>
                  <p className="text-[#575757] dark:text-light/80 text-base text-center">
                    {new Date(comment?.createdAt)?.toLocaleDateString("fa-IR")}
                  </p>
                </td>
                <td>
                  <p className="text-[#575757] dark:text-light/80 text-base text-center">
                    {new Date(comment?.createdAt)?.toLocaleDateString("fa-IR")}
                  </p>
                </td>
                <td className="">
                  {!comment?.isapproved ? (
                    <Button
                      className={`flex-center p-2.5 duration-150 text-white text-xs mx-auto rounded-md bg-[#00B074] hover:bg-[#00B074]/90`}
                      onClick={() =>
                        acceptCommentHandler(comment?._id, fieldId ?? "")
                      }
                    >
                      تایید کامنت
                    </Button>
                  ) : (
                    <Button
                      className={`flex-center p-2.5 duration-150 text-white text-xs mx-auto rounded-md bg-[#FF0000] hover:bg-[#FF0000]/90`}
                      onClick={() =>
                        rejectCommentHandler(comment?._id, fieldId ?? "")
                      }
                    >
                      رد کامنت
                    </Button>
                  )}
                </td>
                <td className="">
                  {!comment?.is_page_show ? (
                    <Button
                      className={`flex-center p-2.5 duration-150 text-white text-xs mx-auto rounded-md bg-[#00B074] hover:bg-[#00B074]/90`}
                      onClick={() =>
                        topCommentHandler(comment?._id, fieldId ?? "")
                      }
                    >
                      افزودن
                    </Button>
                  ) : (
                    <Button
                      className={`flex-center p-2.5 duration-150 text-white text-xs mx-auto rounded-md bg-[#FF0000] hover:bg-[#FF0000]/90`}
                      onClick={() =>
                        topCommentHandler(comment?._id, fieldId ?? "")
                      }
                    >
                      حذف
                    </Button>
                  )}
                </td>
                <td className="">
                  <Button
                    className={`flex-center p-2.5 duration-150 text-white text-xs mx-auto rounded-md bg-secondary-default hover:bg-secondary-default/90`}
                    onClick={() => {
                      setIsOpen(true);
                      setCommentData(comment);
                    }}
                  >
                    مشاهده
                  </Button>
                </td>
              </tr>
            ))}
          </tbody>
        ) : (
          <tbody>
            <tr>
              <td colSpan={4} className="text-center py-10 text-[#EE404C]">
                دیتایی یافت نشد
              </td>
            </tr>
          </tbody>
        )}
      </table>
      {isOpen && (
        <AnswerCommentModal
          setIsOpen={setIsOpen}
          commentData={commentData}
          fieldId={fieldId}
        />
      )}
    </>
  );
}
