"use client";
import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import CustomToast from "@/components/modules/CustomToast";
import * as yup from "yup";
import dynamic from "next/dynamic";
import { blogSchema } from "@/configs/ValidationSchema";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { Controller, SubmitHandler, useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import { ApiError, ApiResponse } from "@/@types/api";
import { AxiosResponse } from "axios";
import { addBlog } from "@/services/blog";
import { Checkbox } from "@/components/ui/Checkbox";
import Image from "next/image";
import { UploadType } from "@/@types/Uploade";

type inputs = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  textarea?: boolean;
  checkbox?: boolean;
  value:
    | "MetaDescription"
    | "MetaTitle"
    | "dec"
    | "href"
    | "keywords"
    | "title"
    | "alt"
    | "show";
};
type InputsType = yup.InferType<typeof blogSchema>;

const BlogEditor = dynamic(() => import("../blog/BlogEditor"), { ssr: false });

export default function AddBlogForm() {
  const [blogContent, setBlogContent] = useState<string>("");
  const [isFileError, setIsFileError] = useState<boolean>(false);
  const [fileError, setFileError] = useState<string | null>(null);
  const [file, setFile] = useState<File[] | null>(null);
  const [uploadProgress, setUploadProgress] = useState<number>(0);

  const queryClient = useQueryClient();

  const inputs: inputs[] = [
    {
      id: 1,
      label: "  متادیسکریپشن :",
      required: true,
      inputType: "text",
      value: "MetaDescription",
    },
    {
      id: 2,
      label: "  متا تایتل :",
      required: true,
      inputType: "text",
      value: "MetaTitle",
    },
    {
      id: 3,
      label: "    عنوان :",
      required: true,
      inputType: "text",
      value: "title",
    },
    {
      id: 4,
      label: "لینک مقاله :",
      required: true,
      inputType: "text",
      value: "href",
    },
    {
      id: 5,
      label: " کلمات کلیدی :",
      required: true,
      inputType: "text",
      value: "keywords",
    },
    {
      id: 6,
      label: "توضیحات عکس(alt) :",
      required: true,
      inputType: "text",
      value: "alt",
    },
    {
      id: 7,
      label: "   توضیحات کوتاه :",
      required: true,
      inputType: "text",
      value: "dec",
      textarea: true,
    },
    {
      id: 8,
      label: "بلاگ فعال",
      required: false,
      inputType: "text",
      checkbox: true,
      value: "show",
    },
  ];

  const {
    register,
    handleSubmit,
    reset,
    control,
    formState: { errors },
  } = useForm<InputsType>({
    resolver: yupResolver(blogSchema),
    defaultValues: {
      MetaDescription: "",
      MetaTitle: "",
      dec: "",
      href: "",
      keywords: "",
      title: "",
      alt: "",
      show: false,
    },
  });

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    UploadType
  >({ mutationFn: addBlog });

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const fileList = e.target.files;
    if (!fileList) return;

    setIsFileError(false);
    setFileError(null);

    const selectedFiles = Array.from(fileList);
    setFile(selectedFiles);
    e.target.value = "";
  };

  const submitHandler: SubmitHandler<InputsType> = (data) => {
    if (!file || file.length === 0) {
      setIsFileError(true);
      setFileError("لطفا عکس خود را انتخاب کنید");
      return;
    }
    if (!blogContent || !blogContent.trim()) {
      CustomToast({
        type: "error",
        text: "لطفا متن مقاله خود را بنویسید",
      });
      return;
    }
    const formData: FormData = new FormData();

    Object.entries(data).forEach(([key, value]) => {
      formData.append(key, String(value));
    });

    file.forEach((f) => {
      formData.append("img", f);
    });
    // formData.append("uid", blogUID);
    // formData.append("id", blogId || "");
    formData.append("text", blogContent);

    mutate(
      { data: formData, onProgress: setUploadProgress },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "مقاله با موفقیت ایجاد شد",
            });
            queryClient.invalidateQueries({ queryKey: ["admin-blogs"] });
            setBlogContent("");
            setFile(null);
            setUploadProgress(0);
            reset({
              MetaDescription: "",
              MetaTitle: "",
              dec: "",
              href: "",
              keywords: "",
              title: "",
              alt: "",
            });
          }
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "دیتا وارد شده معتبر نمی باشد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
          setUploadProgress(0);
        },
      }
    );
  };

  return (
    <form onSubmit={handleSubmit(submitHandler)}>
      <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
        {inputs.map((input) => (
          <div
            key={input.id}
            className={`child:text-[#696464] dark:child:text-light ${
              input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
            }`}
          >
            {!input.checkbox ? (
              <p className="font text-lg">
                {input.label}{" "}
                {input.required ? (
                  <span className="text-error-400">*</span>
                ) : (
                  ""
                )}
              </p>
            ) : null}

            {input.textarea ? (
              <textarea
                className="w-full  mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none"
                placeholder="وارد کنید"
                {...register(input.value)}
                rows={4}
              />
            ) : input.checkbox ? (
              <Controller
                name="show"
                control={control}
                render={({ field }) => (
                  <div className="flex items-center gap-2 h-full mt-5">
                    <Checkbox
                      className="size-6"
                      checked={field.value}
                      onCheckedChange={field.onChange}
                    />
                    <p className="text-lg">مقاله فعال</p>
                  </div>
                )}
              />
            ) : (
              <input
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                type={input.inputType}
                placeholder={
                  input?.value === "keywords"
                    ? "کلمات کلیدی رو با , جدا کنید"
                    : "وارد کنید"
                }
                {...register(input.value)}
              />
            )}

            {errors[input.value] && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors[input.value]?.message}
                </p>
              </div>
            )}
          </div>
        ))}
      </div>
      {/* upload-img */}
      <div className="my-7">
        <p className="font text-lg child:text-[#696464] dark:child:text-light">
          <span className="font text-lg">آپلود عکس :</span>
        </p>
        <label className="flex items-center justify-between w-full sm:w-[180px] py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
          <span className="text-[#9B9B9B]">آپلود عکس</span>
          <input
            className="opacity-0 w-0 h-0"
            type="file"
            onChange={handleFileChange}
          />
          <svg className="w-6 h-6 text-[#9B9B9B]">
            <use href="#file"></use>
          </svg>
        </label>
        {isFileError && fileError && (
          <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
            <svg className="size-5 shrink-0">
              <use href="#danger"></use>
            </svg>
            <p className="text-wrap whitespace-normal">{fileError}</p>
          </div>
        )}
      </div>
      <div className="col-span-1 md:col-span-2 xl:col-span-3 flex flex-wrap gap-2 my-2.5">
        {file?.map((f, index) => (
          <div
            key={index}
            className="relative w-24 h-24 border rounded overflow-hidden"
          >
            <Image
              src={URL.createObjectURL(f)}
              alt={f.name}
              width={96}
              height={96}
              className="w-full h-full object-cover"
            />
            <Button
              className="absolute top-1 right-1 bg-[#FF0000] hover:bg-white hover:text-[#FF0000] text-white w-5 h-5 p-0 m-0 flex items-center justify-center rounded-full"
              onClick={() =>
                setFile((prev) =>
                  prev ? prev.filter((_, i) => i !== index) : null
                )
              }
            >
              <svg>
                <use href="#x-mark"></use>
              </svg>
            </Button>
          </div>
        ))}
      </div>
      <BlogEditor blogContent={blogContent} setBlogContent={setBlogContent} />
      <div className="w-full text-left xl:text-left mt-[23px]">
        <Button
          className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
          type="submit"
          disabled={isPending}
        >
          {isPending ? `در حال آپلود ${uploadProgress} %` : "ایجاد"}
        </Button>
      </div>
    </form>
  );
}
