"use client";
import React, { Dispatch, SetStateAction, useState } from "react";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { updateADminDevStatus } from "@/services/dev";
import { AxiosResponse } from "axios";
import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";

type userStatusModal = {
  setIsOpenUpdateStatusModal: Dispatch<SetStateAction<boolean>>;
  devId: string;
  page: number;
};
type input = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  value: "reason";
  textarea?: boolean;
  ltr?: boolean;
  leftDir?: boolean;
};
type devDataType = {
  reason: string;
  action: "reject" | "approve";
};
export default function AdminDevStatusModal({
  setIsOpenUpdateStatusModal,
  devId,
  page,
}: userStatusModal) {
  const inputs: input[] = [
    {
      id: 1,
      label: " علت رد توسعه دهنده:",
      required: false,
      inputType: "text",
      value: "reason",
    },
  ];
  const [devData, setDevData] = useState<devDataType>({
    reason: "",
    action: "reject",
  });
  const [isError, setIsError] = useState<boolean>(false);
  const queryClient = useQueryClient();
  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    {
      id: string;
      action: string;
      reason: string;
    }
  >({ mutationFn: updateADminDevStatus });

  const clickHandler = () => {
    if (devData?.action === "reject" && !devData.reason) {
      return setIsError(true);
    }
    setIsError(false);
    mutate(
      { id: devId, ...devData },
      {
        onSuccess: () => {
          CustomToast({
            type: "success",
            text: " وضعیت توسعه دهنده با موفقیت تغییر کرد",
          });
          queryClient.invalidateQueries({ queryKey: ["admin-dev", page] });
          setIsOpenUpdateStatusModal(false);
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 404) {
            CustomToast({
              type: "error",
              text: "حساب کاربری پیدا نشد",
            });
          } else if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "شناسه نامعتبر",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };
  return (
    <div className="fixed inset-0 flex items-center justify-center mx-auto w-screen h-screen overflow-hidden animate-fadeIn z-50">
      <div className="w-[350px] md:w-[400px] p-3 bg-light dark:bg-dark-400 rounded-10 z-50">
        <div>
          <p className="font-bold text-base text-[#001A46] dark:text-light">
            تایید حساب توسعه دهنده
          </p>
          <div className="flex items-center mt-4">
            <span className="w-[20%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[80%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <div className="space-y-4 my-2.5">
          <select
            className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none"
            onChange={(e) =>
              setDevData((prev) => ({
                ...prev,
                action: e.target.value === "reject" ? "reject" : "approve",
              }))
            }
          >
            <option value="reject" defaultChecked>
              رد
            </option>
            <option value="approve"> تأیید </option>
          </select>

          {inputs.map((input) => (
            <div
              key={input.id}
              className={"child:text-[#696464] dark:child:text-light"}
            >
              <p className="font text-lg">{input.label}</p>

              <input
                className={`w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm ${
                  input.ltr ? "input__ltr" : ""
                } ${input.leftDir ? "input__left" : ""}`}
                type="text"
                value={devData[input.value]}
                onChange={(e) =>
                  setDevData((prev) => ({ ...prev, reason: e.target.value }))
                }
                placeholder="وارد کنید ..."
              />
              {isError && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">
                    علت رد توسعه دهنده را وارد کنید
                  </p>
                </div>
              )}
            </div>
          ))}
        </div>
        <div className="flex-center gap-2.5 w-full py-2">
          <Button
            className="inline-block flex-center h-10 px-5 bg-secondary-default hover:bg-secondary-default/90 text-base text-light hover:text-light rounded"
            variant="outline"
            onClick={() => {
              setIsOpenUpdateStatusModal(false);
            }}
          >
            لغو
          </Button>
          <Button
            className="inline-block flex-center h-10 px-5 bg-red-600 hover:bg-red-500 text-base text-light rounded"
            onClick={clickHandler}
            disabled={isPending}
          >
            تأیید
          </Button>
        </div>
      </div>

      <div
        className="absolute w-full h-full z-40"
        onClick={() => {
          setIsOpenUpdateStatusModal(false);
        }}
      ></div>
    </div>
  );
}
