"use client";

import React, { useContext, useEffect, useState } from "react";
import { Button } from "@/components/ui/button";
import Image from "next/image";
import Link from "next/link";
import { usePathname, useRouter, useSearchParams } from "next/navigation";
import Auth from "@/components/templates/login/auth";
import { userContext } from "@/app/context/UserProvider";
import { OrderCountContext } from "@/app/context/OrderCountProvider";
import Sidebar from "@/components/templates/login/Sidebar";
import { useQuery } from "@tanstack/react-query";
import { getCategory } from "@/services/category";
import { CategoryType } from "@/@types/categorey";

type SubType = {
  _id: string;
  title: string;
  href: string;
  img: string;
};

type SubCatType = SubType & {
  subsubcategories?: SubType[];
};

type subCategoryType = SubType & {
  subcategories: SubCatType[];
};

export const Header = () => {
  const pathname = usePathname();
  const router = useRouter();
  const searchParams = useSearchParams();
  const refCode = searchParams?.get("refCode");

  const [isLoginModal, setIsLoginModal] = useState<boolean>(false);
  const [isOpenSidebar, setIsOpenSidebar] = useState<boolean>(false);
  const user = useContext(userContext);
  const counterContextData = useContext(OrderCountContext);
  const { data: categories } = useQuery({
    queryKey: ["category"],
    queryFn: getCategory,
    select: (res) => res.data.categorys,
  });
  useEffect(() => {
    if (!user?.userInfo && refCode) {
      setIsLoginModal(true);
    } else {
      setIsLoginModal(false);
    }
  }, [refCode, user]);
  function toggleDarkModeClass() {
    document.documentElement.classList.toggle("dark");
  }
  if (
    pathname.startsWith("/user-panel") ||
    pathname.startsWith("/admin-panel")
  ) {
    return;
  }
  return (
    <header className=" bg-light sm:bg-light-200 sm:dark:bg-dark-300">
      {/* desktop */}
      <div className="container hidden lg:block">
        <div className=" flex items-center justify-between py-2.5 lg:py-7.5">
          <div className="flex items-center">
            <div className="w-10 lg:w-19.5 h-10 lg:h-19.5">
              <Image
                src={"/images/darkobit-logo.svg"}
                width={"78"}
                height={"78"}
                className="w-full h-full"
                alt="لوگو"
              />
            </div>

            <div className="mt-3.5 mr-1 lg:mr-2.5 text-right">
              <Link href="/" className="w-24 lg:w-38.25 h-5 lg:h-8.75">
                <Image
                  src={"/images/logo.svg"}
                  width={"78"}
                  height={"78"}
                  className="w-full h-full"
                  alt="لوگو"
                />
              </Link>
              <p className="hidden lg:inline-block mt-0.75 mr-1 font-peyda font-bold text-sm lg:text-base text-neutral-400 dark:text-light">
                مرجع دانلود فایل
              </p>
            </div>
          </div>
          <div className="flex items-center gap-1 ">
            {user?.isLoading ? (
              <div className="flex items-center justify-center animate-pulse">
                <span className="inline-block w-20 h-9 lg:h-11 bg-neutral-150 border border-light-400 rounded-default "></span>
              </div>
            ) : user?.userInfo !== null ? (
              <>
                <Button
                  className="h-9 lg:h-11 bg-[#222222] hover:bg-[#222222]/90 text-light border border-[#4E4E4E] rounded-default"
                  onClick={() => {
                    router.push("/user-panel");
                  }}
                >
                  پروفایل
                </Button>
                {user?.userInfo.role === "ADMIN" && (
                  <Button
                    className="h-9 lg:h-11 bg-primary-700/90 hover:bg-primary-700/80 text-light border border-primary-700 rounded-default"
                    onClick={() => {
                      router.push("/admin-panel");
                    }}
                  >
                    پنل ادمین
                  </Button>
                )}
              </>
            ) : (
              <Button
                className="h-9 lg:h-11 bg-[#222222] hover:bg-[#222222]/90 text-light border border-[#4E4E4E] rounded-default"
                onClick={() => setIsLoginModal(true)}
              >
                ورود / ثبت نام
              </Button>
            )}

            {/* <Button className="hidden sm:inline-block w-16 lg:w-20 h-9 lg:h-11 bg-secondary-default hover:bg-secondary-default/90 dark:text-light rounded-default">
              ثبت نام
            </Button> */}
            <Button
              className="hidden sm:inline-block flex-center bg-secondary-default hover:bg-secondary-default/90 dark:text-light rounded-default"
              onClick={toggleDarkModeClass}
            >
              <svg
                xmlns="http://www.w3.org/2000/svg"
                fill="none"
                viewBox="0 0 24 24"
                strokeWidth={1.5}
                stroke="currentColor"
                className="size-6"
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  d="M12 3v2.25m6.364.386-1.591 1.591M21 12h-2.25m-.386 6.364-1.591-1.591M12 18.75V21m-4.773-4.227-1.591 1.591M5.25 12H3m4.227-4.773L5.636 5.636M15.75 12a3.75 3.75 0 1 1-7.5 0 3.75 3.75 0 0 1 7.5 0Z"
                />
              </svg>
            </Button>
          </div>
        </div>
        <nav className="hidden lg:flex items-center justify-between px-4 py-2.5 bg-light dark:bg-dark-500 rounded-30">
          <ul className="flex items-center gap-2 xl:gap-3.75 font-peyda font-medium child:cursor-pointer">
            <li className="header__nav bg-primary-default relative group text-dark">
              <Button className="rounded-9 w-8 h-8 dark:bg-black">
                <svg className="w-5 h-5 text-white hover:text-dark shrink-0">
                  <use href="#heart"></use>
                </svg>
              </Button>
              <Link href="#">
                <p className="text-dark group-hover:dark:text-dark">
                  دسته‌بندی‌ها
                </p>
              </Link>
              <svg className="w-4 h-4 mt-1 text-dark group-hover:dark:text-dark">
                <use href="#chevron-double-down"></use>
              </svg>
              <div className="opacity-0 group-hover:opacity-100 hidden group-hover:inline-block absolute top-[42px] -right-8 w-[995px] xl:w-[1250px] p-6 transition-all duration-1000 z-30">
                <div className="w-full overflow-y-auto bg-light dark:bg-dark-500 rounded-10 shadow-md border-neutral-400/80">
                  <ul className="relative w-1/4 bg-muted-foreground/10 p-1 rounded border">
                    {categories?.map((category: CategoryType) => (
                      <li
                        key={category?._id}
                        className="header__nav hover:shadow category__li dark:text-light dark:hover:text-dark"
                      >
                        <Link
                          href={`/category/${category?.title}/${category?._id}`}
                          className="header__nav"
                        >
                          <p>{category.title}</p>
                        </Link>
                        <div className="hidden absolute top-0 right-[220px] xl:right-[280px] w-[500px] xl:w-[600px] h-full">
                          <ul className="w-full grid grid-cols-2 gap-2.5 px-7 py-5">
                            {category.subcategories.map(
                              (sub: subCategoryType) => (
                                <li
                                  key={sub?._id}
                                  className="group relative flex items-center gap-2 hover:text-primary-700 text-sm xl:text-base dark:text-light dark:hover:text-primary-700 "
                                >
                                  <span className="inline-block size-1 bg-primary-700 rounded-full"></span>
                                  <Link
                                    href={`/category/${category?.title}/${category?._id}/${sub?.title}/${sub?._id}`}
                                  >
                                    {sub.title}
                                  </Link>
                                  <div className="hidden absolute top-0 right-[220px] xl:right-[280px] w-[500px] xl:w-[600px] h-full">
                                    <ul className="w-full grid grid-cols-2 gap-2.5 px-7 py-5">
                                      {sub.subcategories?.map(
                                        (subCategory: SubType) => (
                                          <li
                                            key={subCategory?._id}
                                            className="group flex items-center gap-2 hover:text-primary-700 text-sm xl:text-base"
                                          >
                                            <span className="inline-block size-1 bg-primary-700 rounded-full"></span>
                                            <Link href="#">
                                              {subCategory.title}
                                            </Link>
                                          </li>
                                        )
                                      )}
                                    </ul>
                                  </div>
                                </li>
                              )
                            )}
                          </ul>
                        </div>
                      </li>
                    ))}
                  </ul>
                </div>
              </div>
            </li>
            <li className="header__nav group">
              <Link href="/products" className="header__nav group">
                <svg className="w-5 h-5 text-neutral-600 dark:text-neutral-150 dark:group-hover:text-dark hover:text-dark shrink-0">
                  <use href="#puzzle-piece"></use>
                </svg>
                <p className="text-dark dark:text-neutral-150 dark:group-hover:text-dark shrink-0">
                  فروش فایل
                </p>
                {/* <svg className="w-4 h-4 mt-1 text-dark dark:text-neutral-150 dark:group-hover:text-dark shrink-0">
                  <use href="#chevron-down"></use>
                </svg> */}
              </Link>
            </li>
            <li className="header__nav group">
              <Link href="/blogs" className="header__nav group">
                <svg className="w-5 h-5 text-dark dark:text-neutral-150 dark:group-hover:text-dark shrink-0">
                  <use href="#wordpress"></use>
                </svg>
                <p className="text-neutral-600 dark:text-neutral-150 dark:group-hover:text-dark shrink-0">
                  وبلاگ
                </p>
                {/* <svg className="w-4 h-4 mt-1 text-neutral-600 dark:text-neutral-150 dark:group-hover:text-dark hover:text-dark shrink-0 ">
                  <use href="#chevron-down"></use>
                </svg> */}
              </Link>
            </li>

            <li className="header__nav relative group p-3">
              <svg className="w-5 h-5 text-neutral-600 dark:text-neutral-150 dark:group-hover:text-dark hover:text-dark shrink-0">
                <use href="#squares-2x2"></use>
              </svg>
              <p className="text-dark dark:text-neutral-150 dark:group-hover:text-dark shrink-0">
                بیشتر
              </p>
              <svg className="w-4 h-4 mt-1 text-dark dark:text-neutral-150 dark:group-hover:text-dark shrink-0">
                <use href="#chevron-down"></use>
              </svg>
              <div className="absolute top-9 right-0 hidden group-hover:inline-block h-24 p-2 z-30">
                <ul className="bg-light dark:bg-dark-500 p-1 mt-3 rounded-10 shadow-lg border border-neutral-400/10">
                  <li className="header__nav pr-1 child:hover:!text-dark">
                    <Link
                      href="/about-us"
                      className="header__nav pr-1 h-full w-full "
                    >
                      <svg className="w-5 h-5 text-neutral-600 dark:text-neutral-150 dark:group-hover:text-inherit shrink-0">
                        <use href="#puzzle-piece"></use>
                      </svg>
                      <p className="text-dark dark:text-neutral-150 dark:group-hover:text-inherit shrink-0">
                        درباره ما
                      </p>
                    </Link>
                  </li>
                  <li className="header__nav pr-1 child:hover:!text-dark">
                    <Link href="/contact-us" className="header__nav pr-1 ">
                      <svg className="w-5 h-5 text-neutral-600 dark:text-neutral-150 dark:group-hover:text-inherit shrink-0">
                        <use href="#puzzle-piece"></use>
                      </svg>
                      <p className="text-dark dark:text-neutral-150 dark:group-hover:text-inherit shrink-0">
                        تماس با ما
                      </p>
                    </Link>
                  </li>
                </ul>
              </div>
            </li>
          </ul>
          <div>
            <div className=" flex items-center font-yekanbakh">
              <div className="flex items-center">
                <div>
                  <p className="font-medium text-base text-neutral-800 dark:text-neutral-150">
                    021-45697871
                  </p>
                  <p className=" font-medium text-sm-mines ">
                    <span className="text-primary-700">پشتیبانی</span> 24 ساعته
                  </p>
                </div>
                <svg className="size-5 mr-2 text-neutral-800 dark:text-light">
                  <use href="#phone"></use>
                </svg>
              </div>
              <div className="flex items-center gap-2 mr-3.25">
                <div className="flex items-center gap-0.5 px-2.5 py-2 bg-light-500 dark:bg-dark-300 rounded-20">
                  <p className="flex items-center justify-center size-4 pt-0.5 px-1 bg-[#6DA975]/20 text-base text-[#6DA975] rounded-full">
                    0
                  </p>
                  <svg className="size-5 text-[#6DA975]">
                    <use href="#bookmark"></use>
                  </svg>
                </div>
                <Link
                  href={"/orders"}
                  className="flex items-center gap-4 px-3 xl:px-6 py-2.25 bg-dark-400 dark:bg-dark-300 rounded-20"
                >
                  <p className="flex items-center justify-center size-4 pt-0.5 bg-light/20 text-white rounded-full px-1">
                    {counterContextData?.orderCount}
                  </p>
                  <svg className="size-6 text-light">
                    <use href="#shop"></use>
                  </svg>
                </Link>
              </div>
            </div>
          </div>
        </nav>
      </div>
      {/* mobile */}
      <div className="inline-flex items-center justify-between w-full lg:hidden h-[78px] px-[19px] dark:bg-[#222222] shadow-md">
        <Link href={"/"}>
          <Image
            src={"/images/darkobitLogo.svg"}
            alt="عکس لوگو سایت دارکو بیت"
            width={143}
            height={43}
          />
        </Link>

        <div className="flex items-center gap-5">
          <Link
            href={"/orders"}
            className="flex items-center gap-2.5 px-2.25 py-1.5 bg-dark-400 rounded-20"
          >
            <p className="flex items-center justify-center size-4 pt-0.5 bg-light/20 text-white rounded-full px-1">
              {counterContextData?.orderCount}
            </p>
            <svg className="size-6">
              <use href="#shop"></use>
            </svg>
          </Link>
          <svg
            className="size-5 cursor-pointer"
            onClick={() => setIsOpenSidebar((prev) => !prev)}
          >
            <use href="#bars"></use>
          </svg>
        </div>
      </div>
      {isLoginModal && (
        <Auth
          isLoginModal={isLoginModal}
          setIsLoginModal={setIsLoginModal}
          refCode={refCode}
        />
      )}
      <Sidebar
        isOpenSidebar={isOpenSidebar}
        setIsOpenSidebar={setIsOpenSidebar}
        setIsLoginModal={setIsLoginModal}
      />
    </header>
  );
};
