import { file } from "@/@types/file";
import { OrderCountContext } from "@/app/context/OrderCountProvider";
import {
  decreaseCountOrder,
  increaseCountOrder,
  productCounter,
  removeProductOrder,
} from "@/lib/utils";
import Image from "next/image";
import React, { useContext, useState } from "react";
import CustomToast from "./CustomToast";

type OrderCardProps = {
  data: file;
};

export default function OrderCard({ data }: OrderCardProps) {
  const contextData = useContext(OrderCountContext);

  const orderDetails = contextData?.localDate.find(
    (item) => item.productid === data._id
  );

  const orderCount = productCounter(contextData?.localDate ?? [], data._id);
  const [productCont, setProductCount] = useState(
    orderCount === 0 ? 1 : orderCount
  );

  const removeOrder = () => {
    const orderData = {
      productid: data._id,
      price: data.price,
      count: productCont,
    };
    const removeProduct = removeProductOrder(
      contextData?.localDate ?? [],
      orderData
    );
    setProductCount(1);
    contextData?.setLocalDate([...removeProduct]);
    localStorage.setItem("orders", JSON.stringify(removeProduct));
    CustomToast({
      text: "محصول مورد نظر با موفقیت از سبد خرید حذف شد",
      type: "success",
    });
  };

  const increaseHandler = () => {
    if (productCont < 10) {
      setProductCount((prev) => prev + 1);
      const orderNewData = increaseCountOrder(contextData?.localDate ?? [], {
        productid: data._id,
        count: productCont + 1,
        price: data?.price,
      });
      contextData?.setLocalDate([...orderNewData]);
      localStorage.setItem("orders", JSON.stringify(orderNewData));
    }
  };

  const decreaseHandler = () => {
    if (productCont > 1) {
      setProductCount((prev) => prev - 1);
      const orderNewData = decreaseCountOrder(contextData?.localDate ?? [], {
        productid: data._id,
        count: productCont - 1,
        price: data?.price,
      });
      contextData?.setLocalDate(orderNewData);
      localStorage.setItem("orders", JSON.stringify([...orderNewData]));
    }
  };

  return (
    <div className="bg-light py-[22px] md:py-[38px] px-3.25 md:px-10">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-6">
          <div className="size-11 md:size-[89px] shrink-0">
            <Image
              src={`${process.env.NEXT_PUBLIC_BASE_URL}${data?.images[0]}`}
              alt={data?.title}
              width={89}
              height={90}
              className="w-full h-full rounded-10 shrink-0"
            />
          </div>
          <div>
            <p className="text-dark-800 font-peyda text-base md:text-xl">
              {data?.title}
            </p>
            <p className="mt-[26px] text-[#444444] text-xs md:text-base line-clamp-1">
              قیمت :
              <span className="text-dark-800">
                {(data?.price).toLocaleString()} تومان
              </span>
            </p>
          </div>
        </div>
        <div>
          <p>
            <span className="font-peyda font-black text-dark-800 text-base md:text-2xl">
              {(
                +(orderDetails?.count ?? 1) *
                +(orderDetails?.price ?? data.price)
              ).toLocaleString()}
            </span>
            <span className="mr-1.75 font-peyda text-dark-800"> تومان</span>
          </p>
          <div className=" flex items-center mt-2">
            <div className="flex items-center justify-between md:w-[96px]  py-2.5 px-3.5 my-2.5 mx-auto border border-[#E1E1E1] rounded-sm child:text-[#081035]">
              <svg
                className={`size-4 cursor-pointer ${
                  productCont === 10 ? "!text-[#081035]/40" : ""
                }`}
                onClick={increaseHandler}
              >
                <use href="#plus"></use>
              </svg>
              <span className="mx-2.5 select-none font-bold">
                {productCont}
              </span>
              <svg
                className={`size-4 cursor-pointer ${
                  productCont === 1 ? "!text-[#081035]/30" : ""
                }`}
                onClick={decreaseHandler}
              >
                <use href="#mines"></use>
              </svg>
            </div>
            <p
              className="mr-1 md:mr-2.5 font-peyda text-error-400 font-bold text-sm md:text-xl cursor-pointer select-none"
              onClick={removeOrder}
            >
              حذف
            </p>
          </div>
        </div>
      </div>
      {/* <div className="flex flex-row items-center gap-2 w-full mt-2">
        {orderDetails?.selectedOptions?.map((item) => (
          <span
            key={item?._id}
            className="bg-primary-700 p-1.5 rounded-13 text-light text-xs lg:text-base"
          >
            {item?.title}: {Number(item?.value)?.toLocaleString()} تومان
          </span>
        ))}
      </div> */}
      <div className="flex flex-row items-center gap-2 w-full mt-2">
        {data?.customOptions?.map((item) => (
          <span
            key={item?._id}
            className="bg-primary-700 p-1.5 rounded-13 text-light text-xs lg:text-base"
          >
            {item?.title}: {Number(item?.value)?.toLocaleString()} تومان
          </span>
        ))}
      </div>
    </div>
  );
}
