"use client";
import React, { useContext } from "react";
import { Button } from "../ui/button";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import { commentSchema } from "@/configs/ValidationSchema";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { sendComment, sendReplayComment } from "@/services/comment";
import CustomToast from "./CustomToast";
import { AxiosError } from "axios";
import { userContext } from "@/app/context/UserProvider";

type CommentType = {
  message: string;
};
type CommentFormProps = {
  productId: string;
  keyValue: string[];
  textareaRef: React.RefObject<HTMLFormElement>;
  commentId: string;
};
export default function CommentForm({
  productId,
  keyValue,
  textareaRef,
  commentId,
}: CommentFormProps) {
  const userData = useContext(userContext);
  const {
    register,
    handleSubmit,
    reset,
    formState: { errors },
  } = useForm<CommentType>({
    resolver: yupResolver(commentSchema),
    defaultValues: {
      message: "",
    },
  });
  const queryClient = useQueryClient();
  const { mutate, isPending } = useMutation({ mutationFn: sendComment });
  const { mutate: replay_mutate, isPending: replay_isPending } = useMutation({
    mutationFn: sendReplayComment,
  });
  const submitHandler = (data: CommentType) => {
    if (!userData?.userInfo)
      return CustomToast({
        type: "error",
        text: "لطفا ابتدا وارد حساب کاربری خود شوید",
      });

    if (commentId) {
      replay_mutate(
        { fileid: productId, text: data.message, commentId },
        {
          onSuccess: () => {
            CustomToast({
              type: "success",
              text: "نظر شما با موفقیت ارسال شد بعد تایید پشتیبانی انتشار خواهد یافت",
            });
            reset({
              message: "",
            });
            queryClient.invalidateQueries({ queryKey: keyValue });
          },
          onError: (error) => {
            const err = error as AxiosError;
            const statusCode = err?.response?.status;
            if (statusCode === 400) {
              CustomToast({
                type: "error",
                text: "ورودی‌های مورد نیاز (fileid، text) ارائه نشده‌اند یا نامعتبرند",
              });
            } else if (statusCode === 404) {
              CustomToast({
                type: "error",
                text: "فایل یا کامنت پیدا نشد",
              });
            } else {
              CustomToast({
                type: "error",
                text: "خطا در انجام عملیات",
                des: "لطفا دوباره امتحان کنید",
              });
            }
          },
        }
      );
    } else {
      mutate(
        { fileid: productId, text: data.message },
        {
          onSuccess: () => {
            CustomToast({
              type: "success",
              text: "نظر شما با موفقیت ارسال شد بعد تایید پشتیبانی انتشار خواهد یافت",
            });
            reset({
              message: "",
            });
            queryClient.invalidateQueries({ queryKey: keyValue });
          },
          onError: (error) => {
            const err = error as AxiosError;
            const statusCode = err?.response?.status;
            if (statusCode === 400) {
              CustomToast({
                type: "error",
                text: "ورودی‌های مورد نیاز (fileid، text) ارائه نشده‌اند یا نامعتبرند",
              });
            } else if (statusCode === 401) {
              CustomToast({
                type: "error",
                text: "لطفا ابتدا وارد حساب کاربری خودش شوید",
              });
            } else {
              CustomToast({
                type: "error",
                text: "خطا در انجام عملیات",
                des: "لطفا دوباره امتحان کنید",
              });
            }
          },
        }
      );
    }
  };
  return (
    <>
      <form
        onSubmit={handleSubmit(submitHandler)}
        className="grid grid-cols-1 gap-y-7 xl:gap-y-[32px] mt-10"
        ref={textareaRef}
      >
        <div className="col-span-2">
          <p className="font text-lg">
            <span className="font text-lg"> پیام :</span>
            <span className="text-error-400">*</span>
          </p>

          <textarea
            className=" w-full bg-[#EFEFEF] dark:bg-[#535353] p-4 mt-3 text-base-mines rounded-sm outline-none overflow-hidden"
            placeholder="پیام خود را بنویسید ..."
            {...register("message")}
            rows={6}
          />
          {errors.message && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">
                {errors.message?.message}
              </p>
            </div>
          )}
        </div>
        <div className="w-full mt-[17px] text-left">
          <Button
            className="hidden sm:inline-block flex-center w-[165px] h-[53px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
            type="submit"
            disabled={isPending || replay_isPending}
          >
            ارسال
          </Button>
        </div>
      </form>
    </>
  );
}
